/*********************************************************************
*                     SEGGER Microcontroller GmbH                    *
*                        The Embedded Experts                        *
**********************************************************************
*                                                                    *
*       (c) 1995 - 2019 SEGGER Microcontroller GmbH                  *
*                                                                    *
*       Internet: segger.com  Support: support_embos@segger.com      *
*                                                                    *
**********************************************************************
*                                                                    *
*       embOS * Real time operating system for microcontrollers      *
*                                                                    *
*       Please note:                                                 *
*                                                                    *
*       Knowledge of this file may under no circumstances            *
*       be used to write a similar product or a real-time            *
*       operating system for in-house use.                           *
*                                                                    *
*       Thank you for your fairness !                                *
*                                                                    *
**********************************************************************
*                                                                    *
*       OS version: 5.06.1                                           *
*                                                                    *
**********************************************************************

-------------------------- END-OF-HEADER -----------------------------
File    : Main.c
Purpose : Sample program demonstrating usage of SystemView via UART interface
*/

#include "RTOS.h"
#include "BSP.h"
#include "Main.h"
#include "SEGGER_RTT.h"
#include "SEGGER_SYSVIEW.h"

#define _SERVER_HELLO_SIZE        (4)
#define _TARGET_HELLO_SIZE        (4)

static const U8 _abHelloMsg[_TARGET_HELLO_SIZE] = { 'S', 'V', (SEGGER_SYSVIEW_VERSION / 10000), (SEGGER_SYSVIEW_VERSION / 1000) % 10 };  // "Hello" message expected by SysView: [ 'S', 'V', <PROTOCOL_MAJOR>, <PROTOCOL_MINOR> ]

static struct {
  U8         NumBytesHelloRcvd;
  U8         NumBytesHelloSent;
  int        ChannelID;
} _SVInfo;

static struct {
 OS_STACKPTR int StackHP[128];     // Task stacks
 OS_TASK         TCBHP;            // Task control blocks
 OS_STACKPTR int StackLP[128];     // Task stacks
 OS_TASK         TCBLP;            // Task control blocks
} _OSInfo;

/*********************************************************************
*
*       _StartSysView()
*
*  Function description
*    This function starts and initializes a SystemView session,
*    if necessary.
*/
static void _StartSysView(void) {
  int r;

  r = SEGGER_SYSVIEW_IsStarted();
  if (r == 0) {
    SEGGER_SYSVIEW_Start();
  }
}

/*********************************************************************
*
*       _cbOnRx()
*
*  Function description
*    This function is called when the UART receives data.
*/
static void _cbOnRx(U8 Data) {
  if (_SVInfo.NumBytesHelloRcvd < _SERVER_HELLO_SIZE) {  // Not all bytes of <Hello> message received by SysView yet?
    _SVInfo.NumBytesHelloRcvd++;
    goto Done;
  }
  _StartSysView();
  SEGGER_RTT_WriteDownBuffer(_SVInfo.ChannelID, &Data, 1);  // Write data into corresponding RTT buffer for application to read and handle accordingly
Done:
  return;
}

/*********************************************************************
*
*       _cbOnTx()
*
*  Function description
*    This function is called when the UART should transmit data.
*/
static int _cbOnTx(U8* pChar) {
  int r;
  
  if (_SVInfo.NumBytesHelloSent < _TARGET_HELLO_SIZE) {  // Not all bytes of <Hello> message sent to SysView yet?
    *pChar = _abHelloMsg[_SVInfo.NumBytesHelloSent];
    _SVInfo.NumBytesHelloSent++;
    r = 1;
    goto Done;
  }
  r = SEGGER_RTT_ReadUpBufferNoLock(_SVInfo.ChannelID, pChar, 1);
  if (r < 0) {  // Failed to read from up buffer?
    r = 0;
  }
Done:
  return r;
}

/*********************************************************************
*
*       _HPTask()
*/
static void _HPTask(void) {
  while (1) {
    OS_TASK_Delay(50);
    SEGGER_SYSVIEW_PrintfHostEx("High prio", 0);
  }
}

/*********************************************************************
*
*       _LPTask()
*/
static void _LPTask(void) {
  while (1) {
    OS_TASK_Delay(200);
    SEGGER_SYSVIEW_PrintfHostEx("Low prio", 0);
  }
}

/*********************************************************************
*
*       main()
*/
int main(void) {
  OS_Init();    // Initialize embOS
  OS_InitHW();  // Initialize required hardware as well as System View
  BSP_Init();   // Initialize LED ports
  OS_GetCPU();
  //
  // Init UART etc. BEFORE starting embOS, so we can have SystemView from very first embOS actions as well
  //
  _SVInfo.ChannelID = SEGGER_SYSVIEW_GetChannelID();  // Store system view channel ID for later communication
  HIF_UART_Init(500 * 1000, _cbOnTx, _cbOnRx);           // Initialize UART, enable UART interrupts
  //
  // Start embOS
  //
  OS_TASK_CREATE(&_OSInfo.TCBHP, "HP Task", 100, _HPTask, _OSInfo.StackHP);
  OS_TASK_CREATE(&_OSInfo.TCBLP, "LP Task",  50, _LPTask, _OSInfo.StackLP);
  OS_Start();   // Start embOS
  return 0;
}

/*************************** End of file ****************************/
