/*********************************************************************
*                    SEGGER Microcontroller GmbH                     *
*                        The Embedded Experts                        *
**********************************************************************
*                                                                    *
*            (c) 2014 - 2019 SEGGER Microcontroller GmbH             *
*                                                                    *
*       www.segger.com     Support: support@segger.com               *
*                                                                    *
**********************************************************************
*                                                                    *
* All rights reserved.                                               *
*                                                                    *
* Redistribution and use in source and binary forms, with or         *
* without modification, are permitted provided that the following    *
* condition is met:                                                  *
*                                                                    *
* o Redistributions of source code must retain the above copyright   *
*   notice, this condition and the following disclaimer.             *
*                                                                    *
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND             *
* CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,        *
* INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF           *
* MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE           *
* DISCLAIMED. IN NO EVENT SHALL SEGGER Microcontroller BE LIABLE FOR *
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR           *
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT  *
* OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;    *
* OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF      *
* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT          *
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE  *
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH   *
* DAMAGE.                                                            *
*                                                                    *
**********************************************************************

-------------------------- END-OF-HEADER -----------------------------

File    : SEGGER_THUMB_Init.s
Purpose : Runtime initialization and standard library-specific code
          for use with the SEGGER Linker and Standard Library.
          Implementation in ARM Thumb and Thumb-2 assembly.

Additional information:

  This file is designed for use with the SEGGER Linker and Standard Library.
  It does not require any modification.

  However, it can be configured by the following preprocessor definitions:
    APP_ENTRY_POINT
      Define the application entry point.
      If not defined, the application entry point is: main

    ARGSSPACE
      Define the space to reserve for application arguments,
      to be passed by the debugger.
      If not defined, 128 bytes are reserved.

    INITIALIZE_USER_SECTIONS
      If defined, user-supplied function InitializeUserMemorySections is called
      to initialize memory sections which are not iautomatically nitialized.

    __HEAP_SIZE__
      (Defined by project options)
      Define the size of the system heap.

    SKIP_CALL_CTORS_DTORS
      If defined, constructors and destructors are not called.
      (Note: In some CPU Packages UpdateSystemCoreClock added to the list of constructors.)

    FULL_LIBRARY
     If defined then 
       - argc, argv are setup by the debug_getargs.
       - the exit symbol is defined and executes on return from main.
       - the exit symbol calls destructors, atexit functions and then debug_exit.
     If not defined then
       - argc and argv are not initialized.
       - the exit symbol is defined, executes on return from main and loops.
*/

        .syntax unified
        .thumb

/*********************************************************************
*
*       Macros
*
**********************************************************************
*/
/*********************************************************************
*
*       INIT_FUNC
*
*    Declare an init function, used by the SEGGER Linker.
*    Put it into the .segger.init. section
*/
.macro INIT_FUNC Name
        .section .segger.init.\Name, "ax"
        .global \Name
        .balign 2
        .thumb_func
\Name:
.endm

/*********************************************************************
*
*       FUNC
*
*    Declare a global function
*/
.macro FUNC Name
        .global \Name
        .balign 2
        .thumb_func
\Name:
.endm

/*********************************************************************
*
*       HELPER
*
*    Declare a global, weak function, 
*    as used for default library helpers and interrupt handlers.
*/
.macro HELPER Name
        .section .text.\Name, "ax", %progbits
        .global \Name
        .balign 2
        .weak \Name  
        .thumb_func
\Name:
.endm

/*********************************************************************
*
*       JUMPTO
*
*    Implement a jump to a symbol, based on the used instruction set.
*    Required for Cortex-M0/M0+/M1 targets, with limited branch range.
*/
.macro JUMPTO Name
#if defined(__thumb__) && !defined(__thumb2__)
        mov R12, R0
        ldr R0, =\Name
        push {R0}
        mov R0, R12
        pop {PC}
#else
        b \Name
#endif
.endm

/*********************************************************************
*
*       Defines, defaults
*
**********************************************************************
*/
#ifndef APP_ENTRY_POINT
#define APP_ENTRY_POINT main
#endif

#ifndef ARGSSPACE
#define ARGSSPACE 128
#endif  

/*********************************************************************
*
*       Extern symbols
*
**********************************************************************
*/

        .extern APP_ENTRY_POINT
#ifdef INITIALIZE_USER_SECTIONS
        .extern InitializeUserMemorySections
#endif

/*********************************************************************
*
*       Global functions
*
**********************************************************************
*/
        .section .init, "ax"  // _start, exit, and exit_loop need to stay in the given order. Place them into one section ".init".

/*********************************************************************
*
*       _start()
*
*  Function description
*    Runtime initialization.
*    Called from reset handler after system initialization.
*    Initialize data in RAM, initialize heap (if used), 
*    call constructors, setup arguments for main (if used),
*    and call the application entry (i.e. main).
*/
  FUNC _start
        //
        // Go through the initialization table,
        // and initialize data in RAM accordingly.
        //
        ldr R4, =__SEGGER_init_table__
1:  
        ldr R0, [R4]
        adds R4, R4, #4
        tst R0, R0
        beq 2f
        blx R0
        b 1b
2:  
#if !defined(__HEAP_SIZE__) || (__HEAP_SIZE__ != 0)
        //
        // Initialize the heap for use with the standard library.
        //
        ldr R0, = __heap_start__
        ldr R1, = __heap_end__
        subs R1, R1, R0
        cmp R1, #8
        blt 1f
        movs R2, #0
        str R2, [R0]
        adds R0, R0, #4
        str R1, [R0]
1:
#endif

#ifdef INITIALIZE_USER_SECTIONS
        //
        // Initialize additional memory sections,
        // which are not handled by SEGGER Linker.
        //
        ldr R2, =InitializeUserMemorySections
        blx R2
#endif

#ifndef SKIP_CALL_CTORS_DTORS
        //
        // Call the constructors
        //
        ldr R0, =__ctors_start__
        ldr R1, =__ctors_end__
ctor_loop:
        cmp R0, R1
        beq ctor_end
        ldr R2, [R0]
        adds R0, #4
        push {R0-R1}  
        blx R2
        pop {R0-R1}
        b ctor_loop
ctor_end:
#endif
#if defined(FULL_LIBRARY) && (ARGSPACE > 0)
        //
        // Setup arguments for main (argc and argv),
        // taken from the debugger.
        //
        movs R0, #ARGSSPACE
        ldr R1, =args
        ldr R2, =debug_getargs  
        blx R2
        ldr R1, =args
#endif
        //
        // Call the application entry (i.e. main).
        //
        ldr R2, =APP_ENTRY_POINT
        blx R2
        //
        // Fall-through to exit if appliction returns from main.
        //

/*********************************************************************
*
*       exit()
*
*  Function description
*    Application exit.
*    Called explicitly by application, or on return from main.
*    Call destructors and atexit functions of application, 
*    and send application return result to debugger.
*/
        .thumb_func
        .global   exit
        .weak     exit
exit:
#ifdef FULL_LIBRARY  
        //
        // Save application return result
        //
        mov R5, R0
#ifndef SKIP_CALL_CTORS_DTORS
        //
        // Call desctructors
        //
        ldr R0, =__dtors_start__
        ldr R1, =__dtors_end__
dtor_loop:
        cmp R0, R1
        beq dtor_end
        ldr R2, [R0]
        add R0, #4
        push {R0-R1}
        blx R2
        pop {R0-R1}
        b dtor_loop
dtor_end:
#endif
        //
        // Call atexit functions
        //
        ldr R2, =_execute_at_exit_fns
        blx R2
        //
        // Send application return result to debugger.
        // The deubgger might close the debug session or halt the application.
        //
        mov R0, R5
        ldr R2, =debug_exit  
        blx R2
        //
        // Fall-through to endless loop if application continues running.
        //
#endif
        //
        // Loop forever after returning from application entry
        //
exit_loop:
        b exit_loop

/*********************************************************************
*
*       Global functions, required by SEGGER Linker
*
*  Additional information
*    These functions can be used by the SEGGER Linker to initialize
*    sections.
*    Only the functions that are used, are actually linked into the 
*    application.
*
**********************************************************************
*/
/*********************************************************************
*
*       __SEGGER_init_lzss()
*
*  Function description
*    Initialize data in RAM from LZSS compressed initialization image.
*
*  Parameters
*    Destination Address
*    Source Address
*
*  Additional information
*    (1) Placed as function pointer in __SEGGER_Init_Table__ 
*        for blocks that need to be initialized from compressed data.
*        Is always selected when compression is set 'lz77' or 'lzss'.
*          "initialize by copy with packing=lzss"
*        Can be selected when compression is set to Auto and lzss is the 
*        best algorithm, leading to the smallest initialization image.
*          "initialize by copy with packing=auto"
*        Gets called in the init.
*    (2) Optimization:
*        The code below is optimized for size.
*        If startup speed is critical instead,
*        it can be replaced by a faster version.
*    (3) Parameters are passed in memory, pointed to by R4.
*/
  INIT_FUNC __SEGGER_init_lzss
        ldr  R0, [R4]         // Destination
        ldr  R1, [R4, #4]     // Source stream
        adds R4, R4, #8
2:
        ldrb R2, [R1]
        adds R1, R1, #1
        tst R2, R2
        beq  9f               // 0 -> end of table
        cmp R2, #0x80
        bcc  1f               // +ve -> literal run
        //
        // -ve -> copy run
        //
        // R0 = pointer to output stream
        // R1 = pointer to input stream
        // R2 = run length
        // R3 = copy byte
        // R4 = pointer to initialization table
        //
3:
        subs R2, R2, #0x80    // Convert to run length
        beq 10f
        ldrb R3, [R1]         // R3 = first byte of distance
        adds R1, R1, #1
        cmp R3, #0x80
        bcc 5f                // R3 < 128, short run
        subs R3, R3, #0x80    // Adjust to recover true run length high byte
        lsls R3, R3, #8       // Prepare to fuse
        ldrb R5, [R1]         // Extract run length low byte
        adds R1, R1, #1
        adds R3, R3, R5       // Construct run length
5:
        subs R5, R0, R3       // Source of where to copy from
4:
        ldrb R3, [R5]         // Source byte of run
        strb R3, [R0]         // Store to destination
        adds R5, R5, #1
        adds R0, R0, #1
        subs R2, R2, #1
        bne  4b
        b    2b
        //
        // +ve -> literal run
        //
        // R0 = pointer to output stream
        // R1 = pointer to input stream
        // R2 = run length
        // R3 = copy byte
        // R4 = pointer to initialization table
        //
1:
        ldrb R3, [R1]         // source byte of run
        adds R1, R1, #1
        strb R3, [R0]         // store to destination
        adds R0, R0, #1
        subs R2, R2, #1
        bne  1b
        b    2b
9:
        bx LR                 // Done, return.
10:
        b 10b                 // Error, loop forever.

/*********************************************************************
*
*       __SEGGER_init_zero()
*
*  Function description
*    Initialize data in RAM by zeroing it.
*
*  Parameters
*    Destination Address
*    Destination Size
*
*  Additional information
*    (1) Placed as function pointer in __SEGGER_Init_Table__ 
*        for blocks that need to be initialized with 0.
*        Gets called in the init.
*    (2) Optimization:
*        The code below is optimized for size.
*        If startup speed is critical instead,
*        it can be replaced by a faster version.
*    (3) Parameters are passed in memory, pointed to by R4.
*/
  INIT_FUNC __SEGGER_init_zero
        ldr R0, [R4]          // Destination
        ldr R1, [R4, #4]      // Size
        adds R4, R4, #8
        tst R1, R1
        beq 2f
        movs R2, #0
1:
        strb R2, [R0]
        adds R0, R0, #1
        subs R1, R1, #1
        bne 1b
2:
        bx LR

/*********************************************************************
*
*       __SEGGER_init_copy()
*
*  Function description
*    Initialize data in RAM by copy from initialization image.
*
*  Parameters
*    Destination Address
*    Source Address
*    Destination Size
*
*  Additional information
*    (1) Placed as function pointer in __SEGGER_Init_Table__ 
*        for blocks that need to be initialized by copy from flash.
*        Gets called in the init.
*    (2) Optimization:
*        The code below is optimized for size.
*        If startup speed is critical instead,
*        it can be replaced by a faster version.
*    (3) Parameters are passed in memory, pointed to by R4.
*/
  INIT_FUNC __SEGGER_init_copy
        ldr R0, [R4]          // Destination
        ldr R1, [R4, #4]      // Source
        ldr R2, [R4, #8]      // Size
        adds R4, R4, #12
        tst R2, R2
        beq 2f
1:
        ldrb R3, [R1]
        strb R3, [R0]
        adds R0, R0, #1
        adds R1, R1, #1
        subs R2, R2, #1
        bne 1b
2:
        bx LR

/*********************************************************************
*
*       __SEGGER_init_pack()
*
*  Function description
*    Initialize data in RAM from packed initialization image.
*
*  Parameters
*    Destination Address
*    Source Address
*
*  Additional information
*    (1) Placed as function pointer in __SEGGER_Init_Table__ 
*        for blocks that need to be initialized from packed data.
*        Gets called in the init.
*    (2) Optimization:
*        The code below is optimized for size.
*        If startup speed is critical instead,
*        it can be replaced by a faster version.
*    (3) Parameters are passed in memory, pointed to by R4.
*/
  INIT_FUNC __SEGGER_init_pack
        ldr  R0, [R4]         // Destination
        ldr  R1, [R4, #4]     // Source stream
        adds R4, R4, #8
1:
        ldrb R2, [R1]
        adds R1, R1, #1
        cmp  R2, #0x80
        beq  4f
        bcc  3f
        ldrb R3, [R1]         // Byte to replicate
        adds R1, R1, #1
        negs R2, R2
        adds R2, R2, #255
        adds R2, R2, #1
2:
        strb R3, [R0]
        adds R0, R0, #1
        subs R2, R2, #1
        bpl  2b
        b    1b
3:                            // 1+n literal bytes
        ldrb R3, [R1]
        strb R3, [R0]
        adds R0, R0, #1
        adds R1, R1, #1
        subs R2, R2, #1
        bpl  3b
        b    1b
4:
        bx LR

/*********************************************************************
*
*       __SEGGER_init_zpak()
*
*  Function description
*    Initialize data in RAM from zpacked initialization image.
*
*  Parameters
*    Destination Address
*    Source Address
*    Destination Size
*
*  Additional information
*    (1) Placed as function pointer in __SEGGER_Init_Table__ 
*        for blocks that need to be initialized from zpacked data.
*        Gets called in the init.
*    (2) Optimization:
*        The code below is optimized for size.
*        If startup speed is critical instead,
*        it can be replaced by a faster version.
*    (3) Parameters are passed in memory, pointed to by R4.
*/
  INIT_FUNC __SEGGER_init_zpak
        ldr  R0, [R4]         // Destination
        ldr  R1, [R4, #4]     // Source stream
        ldr  R2, [R4, #8]     // Size
        adds R4, R4, #12      // Skip table entries
1:
        ldrb R3, [R1]         // Get control byte from source stream
        adds R1, R1, #1
        movs R6, #8
2:
        movs R5, #0           // Prepare zero filler
        lsrs R3, R3, #1       // Get byte control flag
        bcs  3f               // Carry set -> zero filler
        ldrb R5, [R1]         // Get literal byte from source stream
        adds R1, R1, #1
3:
        strb R5, [R0]         // Store initialization byte
        adds R0, R0, #1
        subs R2, R2, #1       // Size -= 1
        beq  4f               // Exit when destination filled
        subs R6, R6, #1       // Decrement bit count
        bne  2b               // Still within this control byte
        b    1b               // Get next control byte
4:
        bx   LR

/*********************************************************************
*
*       Helper functions, required by standard library
*
**********************************************************************
*/
/*********************************************************************
*
*       __aeabi_read_tp()
*/
HELPER __aeabi_read_tp
        ldr R0, =__tbss_start__-8
        bx LR

/*********************************************************************
*
*       abort()
*/
HELPER abort
        b .

/*********************************************************************
*
*       __assert()
*/
HELPER __assert
        b .

/*********************************************************************
*
*       __aeabi_assert()
*/
HELPER __aeabi_assert
        b .

/*********************************************************************
*
*       __sync_synchronize()
*/
HELPER __sync_synchronize
        bx LR
  
/*********************************************************************
*
*       __getchar()
*/
HELPER __getchar
        JUMPTO debug_getchar

/*********************************************************************
*
*       __putchar()
*/
HELPER __putchar
        JUMPTO debug_putchar

/*********************************************************************
*
*       __open()
*/
HELPER __open
        JUMPTO debug_fopen

/*********************************************************************
*
*       __close()
*/
HELPER __close
        JUMPTO debug_fclose

/*********************************************************************
*
*       __write()
*/
HELPER __write   
        mov R3, R0
        mov R0, R1
        movs R1, #1  
        JUMPTO debug_fwrite

/*********************************************************************
*
*       __read()
*/
HELPER __read  
        mov R3, R0
        mov R0, R1
        movs R1, #1 
        JUMPTO debug_fread

/*********************************************************************
*
*       __seek()
*/
HELPER __seek
        push {R4, LR}
        mov R4, R0
        bl debug_fseek
        cmp R0, #0
        bne 1f
        mov R0, R4
        bl debug_ftell
        pop {R4, PC}
1:
        ldr R0, =-1
        pop {R4, PC}

/*********************************************************************
*
*       Data
*
**********************************************************************
*/
/*********************************************************************
*
*       char __user_locale_name_buffer[];
*/
        .section .bss.__user_locale_name_buffer, "aw", %nobits
        .global __user_locale_name_buffer
        .weak __user_locale_name_buffer
        __user_locale_name_buffer:
        .word 0x0

#if defined(FULL_LIBRARY) && (ARGSPACE > 0)
        //
        // Reserve some space in .bss for arguments for main.
        //
        .bss
args:
        .space ARGSSPACE
#endif

/*************************** End of file ****************************/
