/*********************************************************************
*                     SEGGER Microcontroller GmbH                    *
*                        The Embedded Experts                        *
**********************************************************************
*                                                                    *
*       (c) 1995 - 2021 SEGGER Microcontroller GmbH                  *
*                                                                    *
*       Internet: segger.com  Support: support_embos@segger.com      *
*                                                                    *
**********************************************************************
*                                                                    *
*       embOS * Real time operating system for microcontrollers      *
*                                                                    *
*       Please note:                                                 *
*                                                                    *
*       Knowledge of this file may under no circumstances            *
*       be used to write a similar product or a real-time            *
*       operating system for in-house use.                           *
*                                                                    *
*       Thank you for your fairness !                                *
*                                                                    *
**********************************************************************
*                                                                    *
*       OS version: V5.14.0.0                                        *
*                                                                    *
**********************************************************************

-------------------------- END-OF-HEADER -----------------------------
Purpose : Initializes and handles the hardware for embOS
*/

#include "RTOS.h"
#include "SEGGER_SYSVIEW.h"
#include "stm32l4xx.h"

/*********************************************************************
*
*       Defines
*
**********************************************************************
*/

//
// RCC
//
#define RCC_BASE_ADDR            (0x40021000u)
#define RCC_CFGR                 (*(volatile OS_U32*)(RCC_BASE_ADDR + 0x08u))
#define RCC_APB1ENR1             (*(volatile OS_U32*)(RCC_BASE_ADDR + 0x58u))
#define RCC_APB1ENR2             (*(volatile OS_U32*)(RCC_BASE_ADDR + 0x5Cu))
#define RCC_CCIPR                (*(volatile OS_U32*)(RCC_BASE_ADDR + 0x88u))
#define RCC_CSR                  (*(volatile OS_U32*)(RCC_BASE_ADDR + 0x94u))

#define _RCC_CFGR_HPRE_DIV64     (12u <<  4)
#define _RCC_CFGR_HPRE_MASK      (15u <<  4)
#define _RCC_APB1ENR1_LPTIM1EN   ( 1u << 31)
#define _RCC_APB1ENR2_LPTIM2EN   ( 1u <<  5)
#define _RCC_CCIPR_LPTIM2SEL     ( 1u << 20)
#define _RCC_CCIPR_LPTIM1SEL     ( 1u << 18)

//
// FLASH
//
#define FLASH_BASE_ADDR          (0x40022000u)
#define FLASH_ACR                (*(volatile OS_U32*)(FLASH_BASE_ADDR + 0x00u))

#define _FLASH_ACR_LATENCY_0WS   (0u << 0)
#define _FLASH_ACR_LATENCY_4WS   (4u << 0)
#define _FLASH_ACR_LATENCY_MASK  (7u << 0)

//
// PWR
//
#define PWR_BASE_ADDR            (0x40007000u)
#define PWR_CR1                  (*(volatile OS_U32*)(PWR_BASE_ADDR + 0x00u))
#define PWR_SR2                  (*(volatile OS_U32*)(PWR_BASE_ADDR + 0x14u))

#define _PWR_CR1_VOS_RANGE1      (1u <<  9)
#define _PWR_CR1_VOS_RANGE2      (2u <<  9)
#define _PWR_CR1_VOS_MASK        (3u <<  9)
#define _PWR_SR2_VOSF            (1u << 10)

//
// LPTIM1
//
#define LPTIM1_BASE_ADDR         (0x40007C00u)
#define LPTIM1_ISR               (*(volatile OS_U32*)(LPTIM1_BASE_ADDR + 0x00u))
#define LPTIM1_ICR               (*(volatile OS_U32*)(LPTIM1_BASE_ADDR + 0x04u))
#define LPTIM1_IER               (*(volatile OS_U32*)(LPTIM1_BASE_ADDR + 0x08u))
#define LPTIM1_CFGR              (*(volatile OS_U32*)(LPTIM1_BASE_ADDR + 0x0Cu))
#define LPTIM1_CR                (*(volatile OS_U32*)(LPTIM1_BASE_ADDR + 0x10u))
#define LPTIM1_CMP               (*(volatile OS_U32*)(LPTIM1_BASE_ADDR + 0x14u))
#define LPTIM1_ARR               (*(volatile OS_U32*)(LPTIM1_BASE_ADDR + 0x18u))
#define LPTIM1_CNT               (*(volatile OS_U32*)(LPTIM1_BASE_ADDR + 0x1Cu))
#define LPTIM1_OR                (*(volatile OS_U32*)(LPTIM1_BASE_ADDR + 0x20u))

//
// LPTIM2
//
#define LPTIM2_BASE_ADDR         (0x40009400u)
#define LPTIM2_ISR               (*(volatile OS_U32*)(LPTIM2_BASE_ADDR + 0x00u))
#define LPTIM2_ICR               (*(volatile OS_U32*)(LPTIM2_BASE_ADDR + 0x04u))
#define LPTIM2_IER               (*(volatile OS_U32*)(LPTIM2_BASE_ADDR + 0x08u))
#define LPTIM2_CFGR              (*(volatile OS_U32*)(LPTIM2_BASE_ADDR + 0x0Cu))
#define LPTIM2_CR                (*(volatile OS_U32*)(LPTIM2_BASE_ADDR + 0x10u))
#define LPTIM2_CMP               (*(volatile OS_U32*)(LPTIM2_BASE_ADDR + 0x14u))
#define LPTIM2_ARR               (*(volatile OS_U32*)(LPTIM2_BASE_ADDR + 0x18u))
#define LPTIM2_CNT               (*(volatile OS_U32*)(LPTIM2_BASE_ADDR + 0x1Cu))
#define LPTIM2_OR                (*(volatile OS_U32*)(LPTIM2_BASE_ADDR + 0x20u))

#define _LPTIM_ISR_ARRM          (1u << 1)
#define _LPTIM_ICR_ARRMCF        (1u << 4)

#define _LPTIM_PRESCALER        (0)  // Divide by 1
#define _LPTIM_DIVIDER          (1u << _LPTIM_PRESCALER)
#define _LPTIM_CYCLES_PER_TICK  (OS_TIMER_FREQ / OS_INT_FREQ)
#define _LPTIM_MAX_TICKS        ((OS_TIME)(65536 / _LPTIM_CYCLES_PER_TICK))

/*********************************************************************
*
*       System tick settings
*/
#define OS_TIMER_FREQ             (32000 / _LPTIM_DIVIDER)
#define OS_TICK_FREQ              (1000u)
#define OS_INT_FREQ               (OS_TICK_FREQ)

/*********************************************************************
*
*       embOSView settings
*/
#ifndef   OS_VIEW_IFSELECT
  #define OS_VIEW_IFSELECT  OS_VIEW_IF_JLINK
#endif

#if (OS_VIEW_IFSELECT == OS_VIEW_IF_JLINK)
  #include "JLINKMEM.h"
#elif (OS_VIEW_IFSELECT == OS_VIEW_IF_UART)
  #include "BSP_UART.h"
  #define OS_UART      (0u)
  #define OS_BAUDRATE  (38400u)
#endif

/*********************************************************************
*
*       Static data
*
**********************************************************************
*/
#if (OS_VIEW_IFSELECT == OS_VIEW_IF_JLINK)
  const OS_U32 OS_JLINKMEM_BufferSize = 32u;  // Size of the communication buffer for JLINKMEM
#else
  const OS_U32 OS_JLINKMEM_BufferSize = 0u;   // Buffer not used
#endif

/*********************************************************************
*
*       Local functions
*
**********************************************************************
*/

#if (OS_VIEW_IFSELECT == OS_VIEW_IF_UART)
/*********************************************************************
*
*       _OS_OnRX()
*
*  Function description
*    Callback wrapper function for BSP UART module.
*/
static void _OS_OnRX(unsigned int Unit, unsigned char c) {
  OS_USE_PARA(Unit);
  OS_COM_OnRx(c);
}

/*********************************************************************
*
*       _OS_OnTX()
*
*  Function description
*    Callback wrapper function for BSP UART module.
*/
static int _OS_OnTX(unsigned int Unit) {
  OS_USE_PARA(Unit);
  return (int)OS_COM_OnTx();
}
#endif

/*********************************************************************
*
*       _OS_GetHWTimerCycles()
*
*  Function description
*    Returns the current hardware timer count value.
*
*  Return value
*    Current timer count value.
*/
static unsigned int _OS_GetHWTimerCycles(void) {
  return LPTIM1_CNT;
}

/*********************************************************************
*
*       _OS_GetHWTimer_IntPending()
*
*  Function description
*    Returns if the hardware timer interrupt pending flag is set.
*
*  Return value
*    == 0: Interrupt pending flag not set.
*    != 0: Interrupt pending flag set.
*/
static unsigned int _OS_GetHWTimer_IntPending(void) {
  return LPTIM1_ISR & _LPTIM_ISR_ARRM;
}

/*********************************************************************
*
*       _EnterVoltageRange2()
*/
static void _EnterVoltageRange2(void)  {
  if ((PWR_CR1 & _PWR_CR1_VOS_MASK) != _PWR_CR1_VOS_RANGE2) {
    RCC_CFGR  |= _RCC_CFGR_HPRE_DIV64;      // Set frequency to 1.25MHz
    FLASH_ACR &= ~_FLASH_ACR_LATENCY_MASK;  // Set flash wait states to 0
    PWR_CR1    = (PWR_CR1 & ~ _PWR_CR1_VOS_MASK)
               | _PWR_CR1_VOS_RANGE2;       // Set voltage range to 1.0V
  }
}

/*********************************************************************
*
*       _EnterVoltageRange1()
*/
static void _EnterVoltageRange1(void)  {
  if ((PWR_CR1 & _PWR_CR1_VOS_MASK) != _PWR_CR1_VOS_RANGE1) {
    PWR_CR1    = (PWR_CR1 & ~ _PWR_CR1_VOS_MASK)
               | _PWR_CR1_VOS_RANGE1;     // Set voltage range to 1.2V
    while (PWR_SR2 & _PWR_SR2_VOSF);      // Wait until voltage range is changed
    FLASH_ACR |= _FLASH_ACR_LATENCY_4WS;  // Set flash wait states to 0
    RCC_CFGR  &= ~_RCC_CFGR_HPRE_MASK;    // Set frequency to 80MHz
  }
}

/*********************************************************************
*
*       Global functions
*
**********************************************************************
*/

/*********************************************************************
*
*       LPTIM1_IRQHandler()
*
*  Function description
*    This is the hardware timer exception handler.
*/
void LPTIM1_IRQHandler(void);
void LPTIM1_IRQHandler(void) {
  _EnterVoltageRange1();          // Enter high voltage range and increase CPU frequency
  LPTIM1_ICR = LPTIM_ICR_ARRMCF;  // Clear ARR compare flag
  OS_INT_EnterNestable();
  OS_TICK_Handle();
  OS_INT_LeaveNestable();
}

/*********************************************************************
*
*       OS_InitHW()
*
*  Function description
*    Initialize the hardware required for embOS to run.
*/
void OS_InitHW(void) {
  OS_INT_IncDI();
  //
  // Initialize NVIC vector table offset register if applicable for this device.
  // Might be necessary for RAM targets or application not running from 0x00.
  //
#if (defined(__VTOR_PRESENT) && __VTOR_PRESENT == 1)
  SCB->VTOR = (OS_U32)&__Vectors;
#endif
  //
  // Enable instruction and data cache if applicable for this device
  //
#if (defined(__ICACHE_PRESENT) &&  __ICACHE_PRESENT == 1)
  SCB_EnableICache();
#endif
#if (defined(__DCACHE_PRESENT) &&  __DCACHE_PRESENT == 1)
  SCB_EnableDCache();
#endif
  RCC_CSR      |= (1u <<  0);                    // LSION: Enable LSI
  while ((RCC_CSR & RCC_CSR_LSIRDY) == 0);       // Wait until LSI is stable
  RCC_CCIPR    |= _RCC_CCIPR_LPTIM1SEL;          // LPTIM1SEL: Select LSI
  RCC_APB1ENR1 |= _RCC_APB1ENR1_LPTIM1EN;        // LPTIM1EN: Enable LPTIM1 clock
  NVIC_SetPriority(LPTIM1_IRQn, (1u << __NVIC_PRIO_BITS) - 2u);  // Set the priority higher than the PendSV priority
  NVIC_EnableIRQ(LPTIM1_IRQn);
  LPTIM1_CFGR   = (_LPTIM_PRESCALER <<  9);      // PRESC: Prescaler
  LPTIM1_IER    = (1u <<  1);                    // AARMIE: Enable autoreload match interrupt
  LPTIM1_CR     = (1u <<  0);                    // ENABLE: Enable counter. Needed to write ARR.
  LPTIM1_ARR    = _LPTIM_CYCLES_PER_TICK - 1;    // Reload value
  LPTIM1_CR     = (1u <<  0)                     // ENABLE: Enable counter
                | (1u <<  2);                    // CNTSTRT: Start continuous mode
  //
  // Inform embOS about the timer settings
  //
  {
    OS_SYSTIMER_CONFIG SysTimerConfig = {OS_TIMER_FREQ, OS_INT_FREQ, OS_TIMER_UPCOUNTING, _OS_GetHWTimerCycles, _OS_GetHWTimer_IntPending};
    OS_TIME_ConfigSysTimer(&SysTimerConfig);
  }
  //
  // Configure and initialize SEGGER SystemView
  //
#if (OS_SUPPORT_PROFILE != 0)
  SEGGER_SYSVIEW_Conf();
#endif
  //
  // Initialize communication for embOSView
  //
#if (OS_VIEW_IFSELECT == OS_VIEW_IF_JLINK)
  JLINKMEM_SetpfOnRx(OS_COM_OnRx);
  JLINKMEM_SetpfOnTx(OS_COM_OnTx);
  JLINKMEM_SetpfGetNextChar(OS_COM_GetNextChar);
#elif (OS_VIEW_IFSELECT == OS_VIEW_IF_UART)
  BSP_UART_Init(OS_UART, OS_BAUDRATE, BSP_UART_DATA_BITS_8, BSP_UART_PARITY_NONE, BSP_UART_STOP_BITS_1);
  BSP_UART_SetReadCallback(OS_UART, _OS_OnRX);
  BSP_UART_SetWriteCallback(OS_UART, _OS_OnTX);
#endif
  OS_INT_DecRI();
}

/*********************************************************************
*
*       _EndTicklessMode()
*/
static void _EndTicklessMode(void) {
  if (OS_TICKLESS_IsExpired() == 0) {
    OS_TICKLESS_AdjustTime(LPTIM1_CNT / _LPTIM_CYCLES_PER_TICK);
    LPTIM1_CR = 0;                                // Reset counter
    LPTIM1_CR = (1u << 0);                        // ENABLE: Enable counter
    LPTIM1_ARR = _LPTIM_CYCLES_PER_TICK - 1;      // Configure 1ms timer interrupt
    LPTIM1_CR = (1u << 0) | (1u << 2);            // CNTSTRT: Start continuous mode
  } else {
    OS_TICKLESS_AdjustTime(OS_TICKLESS_GetPeriod());
    LPTIM1_ARR = _LPTIM_CYCLES_PER_TICK - 1;      // Configure 1ms timer interrupt
  }
}

/*********************************************************************
*
*       OS_Idle()
*
*  Function description
*    This code is executed whenever no task, software timer, or
*    interrupt is ready for execution.
*
*  Additional information
*    The idle loop does not have a stack of its own, therefore no
*    functionality should be implemented that relies on the stack
*    to be preserved.
*/
void OS_Idle(void) {
  OS_TIME IdleTicks;

  OS_INT_IncDI();
  IdleTicks = OS_TICKLESS_GetNumIdleTicks();                // Get idle period
  if (IdleTicks > 1) {                                      // Start tickless mode only if idle period >1
    if (IdleTicks > _LPTIM_MAX_TICKS) {                     // Make sure the timer can ahndle the idle period
      IdleTicks = _LPTIM_MAX_TICKS;                         // Else, enter tickless mode for the maximum possible period
    }
    LPTIM1_CR = 0;                                          // Reset counter
    LPTIM1_CR = (1u << 0);                                  // ENABLE: Enable timer
    LPTIM1_ARR = (IdleTicks * _LPTIM_CYCLES_PER_TICK) - 1;  // Configure 1ms timer interrupt
    LPTIM1_CR = (1u << 0) | (1u << 2);                      // CNTSTRT: Start counter in continuous mode
    OS_TICKLESS_Start(IdleTicks, _EndTicklessMode);         // Start tickless mode
  }
  OS_INT_DecRI();
  _EnterVoltageRange2();                                    // Enter voltage range 2, decrease CPU frequency
  while (1) {
    __WFI();
  }
}

/*********************************************************************
*
*       Optional communication with embOSView
*
**********************************************************************
*/

/*********************************************************************
*
*       OS_COM_Send1()
*
*  Function description
*    Sends one character.
*/
void OS_COM_Send1(OS_U8 c) {
#if (OS_VIEW_IFSELECT == OS_VIEW_IF_JLINK)
  JLINKMEM_SendChar(c);
#elif (OS_VIEW_IFSELECT == OS_VIEW_IF_UART)
  BSP_UART_Write1(OS_UART, c);
#elif (OS_VIEW_IFSELECT == OS_VIEW_DISABLED)
  OS_USE_PARA(c);          // Avoid compiler warning
  OS_COM_ClearTxActive();  // Let embOS know that Tx is not busy
#endif
}

/*************************** End of file ****************************/
