/*********************************************************************
*                     SEGGER Microcontroller GmbH                    *
*                        The Embedded Experts                        *
**********************************************************************
*                                                                    *
*       (c) 1995 - 2020 SEGGER Microcontroller GmbH                  *
*                                                                    *
*       Internet: segger.com  Support: support_embos@segger.com      *
*                                                                    *
**********************************************************************
*                                                                    *
*       embOS * Real time operating system for microcontrollers      *
*                                                                    *
*       Please note:                                                 *
*                                                                    *
*       Knowledge of this file may under no circumstances            *
*       be used to write a similar product or a real-time            *
*       operating system for in-house use.                           *
*                                                                    *
*       Thank you for your fairness !                                *
*                                                                    *
**********************************************************************
*                                                                    *
*       OS version: V5.10.2.0                                        *
*                                                                    *
**********************************************************************

-------------------------- END-OF-HEADER -----------------------------

File      : SEGGER_ARM_Startup.s
Purpose   : Generic runtime init startup code for ARM CPUs running 
            in ARM mode.
            Designed to work with the SEGGER linker to produce 
            smallest possible executables.
            
            This file does not normally require any customization.

Additional information:
  Preprocessor Definitions
    FULL_LIBRARY
      If defined then 
        - argc, argv are setup by the debug_getargs.
        - the exit symbol is defined and executes on return from main.
        - the exit symbol calls destructors, atexit functions and then debug_exit.
    
      If not defined then
        - argc and argv are not valid (main is assumed to not take parameters)
        - the exit symbol is defined, executes on return from main and loops
*/

        .syntax unified  

/*********************************************************************
*
*       Macros
*
**********************************************************************
*/

#if defined(__ARM_ARCH_4T__)
.macro blx reg
  mov lr, pc
  bx \reg
.endm
#endif

/*********************************************************************
*
*       Defines, configurable
*
**********************************************************************
*/

#ifndef   APP_ENTRY_POINT
  #define APP_ENTRY_POINT main
#endif

#ifndef   ARGSSPACE
  #define ARGSSPACE 128
#endif

/*********************************************************************
*
*       Externals
*
**********************************************************************
*/
        .extern APP_ENTRY_POINT     // typically main
        .extern __low_level_init

/*********************************************************************
*
*       Global functions
*
**********************************************************************
*/
/*********************************************************************
*
*       _start
*
*  Function description
*    Entry point for the startup code. 
*    Usually called by the reset handler.
*    Performs all initialisation, based on the entries in the 
*    linker-generated init table, then calls main().
*    It is device independent, so there should not be any need for an 
*    end-user to modify it.
*/
        .weak _start
        .global __start
        .section .init, "ax"
        .type _start, function
        .code 32
        .balign 4
_start:
__start:
        //
        // Setup Stacks
        //
        mrs     R0, CPSR
        bic     R0, R0, #0x1F
        //
        orr     R1, R0, #0x1B           // Setup CPSR for undefined mode
        msr     CPSR_cxsf, R1           // Put CPU in mode
        ldr     SP, =__stack_und_end__  // Load mode's stack pointer
        bic     SP, SP, #0x7            // Align stack pointer if necessary
        orr     R1, R0, #0x17           // Setup CPSR for abort mode
        msr     CPSR_cxsf, R1           // Put CPU in mode
        ldr     SP, =__stack_abt_end__  // Load mode's stack pointer
        bic     SP, SP, #0x7            // Align stack pointer if necessary
        orr     R1, R0, #0x12           // Setup CPSR for IRQ mode
        msr     CPSR_cxsf, R1           // Put CPU in mode
        ldr     SP, =__stack_irq_end__  // Load mode's stack pointer
        bic     SP, SP, #0x7            // Align stack pointer if necessary
        orr     R1, R0, #0x11           // Setup CPSR for FIQ mode
        msr     CPSR_cxsf, R1           // Put CPU in mode
        ldr     SP, =__stack_fiq_end__  // Load mode's stack pointer
        bic     SP, SP, #0x7            // Align stack pointer if necessary
        orr     R1, R0, #0x13           // Setup CPSR for supervisor mode
        msr     CPSR_cxsf, R1           // Put CPU in mode
        ldr     SP, =__stack_svc_end__  // Load mode's stack pointer
        bic     SP, SP, #0x7            // Align stack pointer if necessary
#ifdef SUPERVISOR_START
        //
        // Application to be started in supervisor mode
        // Setup user mode stack
        //
        ldr     R1, =__stack_end__
        bic     R1, R1, #0x7            // Align stack pointer if necessary
        mov     R2, SP                  // Load current stack pointer to R2
        stmfd   R2!, {R1}               // Store user mode SP on supervisor stack
        ldmfd   R2, {SP}^               // Load user mode stack pointer (without switching to user mode)
#else
        //
        // Application to be started in system mode
        // Switch mode and setup stack
        //
        orr     R1, R0, #0x1F           // Setup CPSR for system mode
        msr     CPSR_cxsf, R1           // Put CPU in mode
        ldr     SP, =__stack_end__      // Load system mode stack pointer
        bic     SP, SP, #0x7            // Align stack pointer if necessary
#endif
        // 
        // Call __low_level_init to initialize hardware
        // before calling c-standard startup
        //
        bl      __low_level_init
        //
        // Call linker init functions which in turn performs the following:
        // * Perform segment init
        // * Perform heap init (if used)
        // * Call constructors of global Objects (if any exist)
        //
        ldr     R4, =__SEGGER_init_table__      // Set table pointer to start of initialization table
__SEGGER_init_run_loop: 
        ldr     R0, [R4]                        // Get next initialization function from table
        adds    R4, R4, #4                      // Increment table pointer to point to function arguments
        blx     R0                              // Call initialization function
        b       __SEGGER_init_run_loop
        //
       .code 32
       .type __SEGGER_init_done, function
       .global __SEGGER_init_done
__SEGGER_init_done:
        //
        // Time to call main(), the application entry point.
        //
#ifndef FULL_LIBRARY
        //
        // In a real embedded application ("Free-standing environment"), 
        // main() does not get any arguments,
        // which means it is not necessary to init R0 and R1.
        //
        bl      APP_ENTRY_POINT                 // Call to application entry point (usually main())
        //
        // Fall-through to exit if main ever returns.
        // 
        .global exit
        .type exit, function
exit:
        //
        // In a free-standing environment, if returned from application:
        // Loop forever.
        //
        b       exit
#else
        //
        // In a hosted environment, 
        // we need to load R0 and R1 with argc and argv, in order to handle 
        // the command line arguments.
        // This is required for some programs running under control of a 
        // debugger, such as automated tests.
        //
        mov     R0, #ARGSSPACE
        ldr     R1, =args
        bl      SEGGER_SEMIHOST_GetArgs
        ldr     R1, =args
        bl      APP_ENTRY_POINT                 // Call to application entry point (usually main())
        //
        // Fall-through to exit if main ever returns.
        // 
        .code 32
        .global exit
        .type exit, function
exit:
        //
        // In a hosted environment exit gracefully, by
        // saving the return value,
        // calling destructurs of global objects, 
        // calling registered atexit functions, 
        // and notifying the host/debugger.
        //
        mov     R5, R0                  // Save the exit parameter/return result
        //
        // Call destructors
        //
        ldr     R0, =__dtors_start__    // Pointer to destructor list
        ldr     R1, =__dtors_end__
dtorLoop:
        cmp     R0, R1
        beq     dtorEnd                 // Reached end of destructor list? => Done
        ldr     R2, [R0], #+4           // Load current destructor address into R2 and increment pointer
        push    {R0-R1}                 // Save R0 and R1
        blx     R2                      // Call destructor
        pop     {R0-R1}                 // Restore R0 and R1
        b       dtorLoop
dtorEnd:
        //
        // Call atexit functions
        //
        bl      __SEGGER_RTL_execute_at_exit_fns
        //
        // Call debug_exit with return result/exit parameter
        //
        mov     R0, R5
        bl      SEGGER_SEMIHOST_Exit
        //
        // If execution is not terminated, loop forever
        //
exit_loop:
        b       exit_loop // Loop forever.
#endif

#ifdef FULL_LIBRARY
        .bss
args:
        .space ARGSSPACE
#endif

/*************************** End of file ****************************/
