/*********************************************************************
*                     SEGGER Microcontroller GmbH                    *
*                        The Embedded Experts                        *
**********************************************************************
*                                                                    *
*       (c) 1995 - 2020 SEGGER Microcontroller GmbH                  *
*                                                                    *
*       Internet: segger.com  Support: support_embos@segger.com      *
*                                                                    *
**********************************************************************
*                                                                    *
*       embOS * Real time operating system for microcontrollers      *
*                                                                    *
*       Please note:                                                 *
*                                                                    *
*       Knowledge of this file may under no circumstances            *
*       be used to write a similar product or a real-time            *
*       operating system for in-house use.                           *
*                                                                    *
*       Thank you for your fairness !                                *
*                                                                    *
**********************************************************************
*                                                                    *
*       OS version: V5.10.2.0                                        *
*                                                                    *
**********************************************************************

-------------------------- END-OF-HEADER -----------------------------

File      : ARM_Startup.s
Purpose   : Generic startup and exception handlers for ARM devices.

Additional information:
  Preprocessor Definitions
    __FLASH_BUILD
      If defined,
        additional code can be executed in Reset_Handler,
        which might be needed to run from Flash but not to run from RAM.

    __SOFTFP__
      Defined by the build system.
      If not defined, the FPU is enabled for floating point operations.
*/

#if defined(__ARM_ARCH_4T__)
.macro blx reg
  mov lr, pc
  bx \reg
.endm
#endif

        .syntax unified

/*********************************************************************
*
*       Externals, code
*
**********************************************************************
*/
        .extern IRQ_Handler  // embOS low level irq handler

/*********************************************************************
*
*       Global data
*
**********************************************************************
*/
/*********************************************************************
*
*  Setup of the exception table
*
*/
        .section .vectors, "ax"
        .code 32
        .balign 4
        .global _vectors
_vectors:
        ldr     PC, =Reset_Handler
        ldr     PC, =undef_handler
        ldr     PC, =swi_handler
        ldr     PC, =pabort_handler
        ldr     PC, =dabort_handler
        nop
        ldr     PC, =IRQ_Handler
        ldr     PC, =fiq_handler

/*********************************************************************
*
*       Global functions
*
**********************************************************************
*/
/*********************************************************************
*
*       Reset_Handler
*
*  Function description
*    Exception handler for reset.
*/
        .section .init, "ax"
        .code 32
        .balign 4
        .global reset_handler
        .global Reset_Handler
        .equ reset_handler, Reset_Handler
        .type Reset_Handler, function
Reset_Handler:
  /****************************************************************************
   * TODO: Configure target system.                                           *
   ****************************************************************************/
#ifdef __FLASH_BUILD
  /****************************************************************************
   * TODO: Put any FLASH build configuation specific code here                *
   ****************************************************************************/
#endif

//==================================================================
// Disable caches, MMU and branch prediction in case they were left enabled
// from an earlier run. This does not need to be done from a cold reset
//==================================================================

        mrc     P15, 0, R0, C1, C0, 0       // Read CP15 System Control register
        bic     R0, R0, #(0x1 << 12)        // Clear I bit 12 to disable I Cache
        bic     R0, R0, #(0x1 <<  2)        // Clear C bit  2 to disable D Cache
        bic     R0, R0, #0x1                // Clear M bit  0 to disable MMU
        bic     R0, R0, #(0x1 << 11)        // Clear Z bit 11 to disable branch prediction
        bic     R0, R0, #(0x1 << 13)        // Clear V bit 13 to disable high vector table
        bic     R0, R0, #(0x1 << 28)        // Clear TRE bit to disable TEX remap
        mcr     P15, 0, R0, C1, C0, 0       // Write value back to CP15 System Control register

        mov     R0, #0x2000                 // Disable L2 cache
        movt    R0, #0xF8F0
        mov     R1, #0x00
        str     R1, [R0, #0]

        ldr     R0, =_vectors               // Load exception table base address to R0
        mcr     P15, 0, R0, C12, C0, 0      // Write address to CP15.VBAR register.

//==================================================================
// Enable RAM
//==================================================================
        //
        // After reset, only 192KB of 256KB iRAM are continuously mapped to 0x00000000
        // Map all iRAM to 0x00000000, so that whole RAM range can be used in the linker file
        //
        // REG_SLCR_UNLOCK  = SCLR_UNLOCK_KEY;
        // REG_SLCR_OCM_CFG = (1u << 4);  // [4] seems to be set and must kept set it seems. Doc does not say anything about this bit...
        // REG_SLCR_LOCK    = SCLR_LOCK_KEY;
        //
        ldr     R3, =0xF8000000
        movw    R2, #0xDF0D
        str     R2, [R3, #8]        // REG_SLCR_UNLOCK = SCLR_UNLOCK_KEY;
        movs    R2, #16             // REG_SLCR_OCM_CFG = (1u << 4); [4] seems to be set and must kept set it seems. Doc does not say anything about this bit...
        str     R2, [R3, #0x910]
        movw    R2, #0x767B
        str     R2, [R3, #4]        // REG_SLCR_LOCK = SCLR_LOCK_KEY;

//==================================================================
// Invalidate Data and Instruction TLBs and branch predictor
//==================================================================

        mov     R0, #0
        mcr     P15, 0, R0, C8, C7, 0      // I-TLB and D-TLB invalidation
        mcr     P15, 0, R0, C7, C5, 6      // BPIALL - Invalidate entire branch predictor array

//==================================================================
// Cache Invalidation code for Cortex-A9
//==================================================================

        // Invalidate L1 Instruction Cache

        mrc     P15, 1, R0, C0, C0, 1      // Read Cache Level ID Register (CLIDR)
        tst     R0, #0x3                   // Harvard Cache?
        mov     R0, #0                     // SBZ
        mcrne   P15, 0, R0, C7, C5, 0      // ICIALLU - Invalidate instruction cache and flush branch target cache

        // Invalidate Data/Unified Caches

        mrc     P15, 1, R0, C0, C0, 1      // Read CLIDR
        ands    R3, R0, #0x07000000        // Extract coherency level
        mov     R3, R3, LSR #23            // Total cache levels << 1
        beq     Finished                   // If 0, no need to clean

        mov     R10, #0                    // R10 holds current cache level << 1
Loop1:  add     R2, R10, R10, LSR #1       // R2 holds cache "Set" position
        mov     R1, R0, LSR R2             // Bottom 3 bits are the Cache-type for this level
        and     R1, R1, #7                 // Isolate those lower 3 bits
        cmp     R1, #2
        blt     Skip                       // No cache or only instruction cache at this level

        mcr     P15, 2, R10, C0, C0, 0     // Write the Cache Size selection register
        isb                                // isb to sync the change to the CacheSizeID reg
        mrc     P15, 1, R1, C0, C0, 0      // Reads current Cache Size ID register
        and     R2, R1, #7                 // Extract the line length field
        add     R2, R2, #4                 // Add 4 for the line length offset (log2 16 bytes)
        ldr     R4, =0x3FF
        ands    R4, R4, R1, LSR #3         // R4 is the max number on the way size (right aligned)
        clz     R5, R4                     // R5 is the bit position of the way size increment
        ldr     R7, =0x7FFF
        ands    R7, R7, R1, LSR #13        // R7 is the max number of the index size (right aligned)

Loop2:  mov     R9, R4                     // R9 working copy of the max way size (right aligned)

Loop3:  orr     R11, R10, R9, LSL R5       // Factor in the Way number and cache number into R11
        orr     R11, R11, R7, LSL R2       // Factor in the Set number
        mcr     P15, 0, R11, C7, C6, 2     // Invalidate by Set/Way
        subs    R9, R9, #1                 // Decrement the Way number
        bge     Loop3
        subs    R7, R7, #1                 // Decrement the Set number
        bge     Loop2
Skip:   add     R10, R10, #2               // increment the cache number
        cmp     R3, R10
        bgt     Loop1

Finished:

//===================================================================
// Cortex-A9 MMU Configuration
// Set translation table base
//===================================================================

        //
        // Cortex-A9 supports two translation tables
        // Configure translation table base (TTB) control register cp15,c2
        // to a value of all zeros, indicates we are using TTB register 0.
        // We don't initialze the TTB here, this is done in __low_level_init() in RTOSInit-C-module
        //
        mov     R0,#0x0
        mcr     P15, 0, R0, C2, C0, 2

#if !defined(__SOFTFP__)
//==================================================================
// Enable VFP/NEON unit
//==================================================================
        //
        // Enable access to co-processor cp10 and cp11
        // for both kernel and user access
        //
        mrc     P15, 0, R0, C1, C0, 2
        orr     R0, R0, #(0xF << 20)       // Kernel+user access to cp11 and cp10
        mcr     P15, 0, R0, C1, C0, 2      // Set co-processor access bits

        //
        // Enable VFP co-processor by setting EN bit in FPEXC register
        //
        mov     R0, #(1 << 30)             // EN (1=enable VFPLite and NEON coprocessor)
        vmsr    FPEXC, R0                  // Set FPEXC register
#endif
        //
        // Jump to runtime initialization, which calls main().
        //
        b _start

/*********************************************************************
*
*       Default exception handlers
*
*  These are declared weak symbols so the can be redefined in user code.
*
**********************************************************************
*/
        .section .init, "ax"
        .code 32
        .balign 4

        .global undef_handler
        .weak undef_handler
        .type undef_handler, function
undef_handler:
        b       undef_handler

        .global swi_handler
        .weak swi_handler
        .type swi_handler, function
swi_handler:
        b       swi_handler

        .global pabort_handler
        .weak pabort_handler
        .type pabort_handler, function
pabort_handler:
        b       pabort_handler

        .global dabort_handler
        .weak dabort_handler
        .type dabort_handler, function
dabort_handler:
        b       dabort_handler

        .global irq_handler
        .weak irq_handler
        .type irq_handler, function
irq_handler:
        b       irq_handler

        .global fiq_handler
        .weak fiq_handler
        .type fiq_handler, function
fiq_handler:
        b       fiq_handler

/*************************** End of file ****************************/
