/*********************************************************************
*                    SEGGER Microcontroller GmbH                     *
*                        The Embedded Experts                        *
**********************************************************************
*                                                                    *
*            (c) 2014 - 2020 SEGGER Microcontroller GmbH             *
*                                                                    *
*       www.segger.com     Support: support@segger.com               *
*                                                                    *
**********************************************************************
*                                                                    *
* All rights reserved.                                               *
*                                                                    *
* Redistribution and use in source and binary forms, with or         *
* without modification, are permitted provided that the following    *
* condition is met:                                                  *
*                                                                    *
* - Redistributions of source code must retain the above copyright   *
*   notice, this condition and the following disclaimer.             *
*                                                                    *
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND             *
* CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,        *
* INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF           *
* MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE           *
* DISCLAIMED. IN NO EVENT SHALL SEGGER Microcontroller BE LIABLE FOR *
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR           *
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT  *
* OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;    *
* OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF      *
* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT          *
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE  *
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH   *
* DAMAGE.                                                            *
*                                                                    *
**********************************************************************

-------------------------- END-OF-HEADER -----------------------------

File      : TMPM4NR_Vectors.s
Purpose   : Exception and interrupt vectors for TMPM4NR devices.

Additional information:
  Preprocessor Definitions
    __NO_EXTERNAL_INTERRUPTS
      If defined,
        the vector table will contain only the internal exceptions
        and interrupts.
    __VECTORS_IN_RAM
      If defined,
        an area of RAM, large enough to store the vector table,
        will be reserved.

    __OPTIMIZATION_SMALL
      If defined,
        all weak definitions of interrupt handlers will share the
        same implementation.
      If not defined,
        all weak definitions of interrupt handlers will be defined
        with their own implementation.
*/
        .syntax unified

/*********************************************************************
*
*       Macros
*
**********************************************************************
*/

//
// Directly place a vector (word) in the vector table
//
.macro VECTOR Name=
        .section .vectors, "ax"
        .code 16
        .word \Name
.endm

//
// Declare an exception handler with a weak definition
//
.macro EXC_HANDLER Name=
        //
        // Insert vector in vector table
        //
        .section .vectors, "ax"
        .word \Name
        //
        // Insert dummy handler in init section
        //
        .section .init.\Name, "ax"
        .thumb_func
        .weak \Name
        .balign 2
\Name:
        1: b 1b   // Endless loop
.endm

//
// Declare an interrupt handler with a weak definition
//
.macro ISR_HANDLER Name=
        //
        // Insert vector in vector table
        //
        .section .vectors, "ax"
        .word \Name
        //
        // Insert dummy handler in init section
        //
#if defined(__OPTIMIZATION_SMALL)
        .section .init, "ax"
        .weak \Name
        .thumb_set \Name,Dummy_Handler
#else
        .section .init.\Name, "ax"
        .thumb_func
        .weak \Name
        .balign 2
\Name:
        1: b 1b   // Endless loop
#endif
.endm

//
// Place a reserved vector in vector table
//
.macro ISR_RESERVED
        .section .vectors, "ax"
        .word 0
.endm

//
// Place a reserved vector in vector table
//
.macro ISR_RESERVED_DUMMY
        .section .vectors, "ax"
        .word Dummy_Handler
.endm

/*********************************************************************
*
*       Externals
*
**********************************************************************
*/
        .extern __stack_end__
        .extern Reset_Handler
        .extern HardFault_Handler

/*********************************************************************
*
*       Global functions
*
**********************************************************************
*/

/*********************************************************************
*
*  Setup of the vector table and weak definition of interrupt handlers
*
*/
        .section .vectors, "ax"
        .code 16
        .balign 512
        .global _vectors
_vectors:
        //
        // Internal exceptions and interrupts
        //
        VECTOR __stack_end__
        VECTOR Reset_Handler
        EXC_HANDLER NMI_Handler
        VECTOR HardFault_Handler
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
        EXC_HANDLER SVC_Handler
        ISR_RESERVED
        ISR_RESERVED
        EXC_HANDLER PendSV_Handler
        EXC_HANDLER SysTick_Handler
        //
        // External interrupts
        //
#ifndef __NO_EXTERNAL_INTERRUPTS
        ISR_HANDLER INT00_IRQHandler          
        ISR_HANDLER INT01_IRQHandler          
        ISR_HANDLER INT02_IRQHandler          
        ISR_HANDLER INT03_IRQHandler          
        ISR_HANDLER INT04_IRQHandler          
        ISR_HANDLER INT05_IRQHandler          
        ISR_HANDLER INT06_IRQHandler          
        ISR_HANDLER INT07_IRQHandler          
        ISR_HANDLER INT08_IRQHandler          
        ISR_HANDLER INT09_IRQHandler          
        ISR_HANDLER INT10_IRQHandler          
        ISR_HANDLER INT11_IRQHandler          
        ISR_HANDLER INT12_IRQHandler          
        ISR_HANDLER INT13_IRQHandler          
        ISR_HANDLER INT14_IRQHandler          
        ISR_HANDLER INT15_IRQHandler          
        ISR_HANDLER INTRTC_IRQHandler         
        ISR_HANDLER INTCEC0RX_IRQHandler      
        ISR_HANDLER INTCEC0TX_IRQHandler      
        ISR_HANDLER INTISDA_IRQHandler        
        ISR_HANDLER INTISDB_IRQHandler        
        ISR_HANDLER INTISDC_IRQHandler        
        ISR_HANDLER INTRMC0_IRQHandler        
        ISR_HANDLER INTRMC1_IRQHandler        
        ISR_HANDLER INTLTTMR0_IRQHandler      
        ISR_HANDLER INTHDMAATC_IRQHandler     
        ISR_HANDLER INTHDMAAERR_IRQHandler    
        ISR_HANDLER INTHDMABTC_IRQHandler     
        ISR_HANDLER INTHDMABERR_IRQHandler    
        ISR_HANDLER INTMDMAATC_IRQHandler     
        ISR_HANDLER INTT32A00_A_CT_IRQHandler 
        ISR_HANDLER INTT32A00_B_C01_CPC_IRQHandler
        ISR_HANDLER INTT32A01_A_CT_IRQHandler 
        ISR_HANDLER INTT32A01_B_C01_CPC_IRQHandler
        ISR_HANDLER INTT32A02_A_CT_IRQHandler 
        ISR_HANDLER INTT32A02_B_C01_CPC_IRQHandler
        ISR_HANDLER INTT32A03_A_CT_IRQHandler 
        ISR_HANDLER INTT32A03_B_C01_CPC_IRQHandler
        ISR_HANDLER INTT32A04_A_CT_IRQHandler 
        ISR_HANDLER INTT32A04_B_C01_CPC_IRQHandler
        ISR_HANDLER INTT32A05_A_CT_IRQHandler 
        ISR_HANDLER INTT32A05_B_C01_CPC_IRQHandler
        ISR_HANDLER INTT32A06_A_CT_IRQHandler 
        ISR_HANDLER INTT32A06_B_C01_CPC_IRQHandler
        ISR_HANDLER INTT32A07_A_CT_IRQHandler 
        ISR_HANDLER INTT32A07_B_C01_CPC_IRQHandler
        ISR_HANDLER INTT32A08_A_CT_IRQHandler 
        ISR_HANDLER INTT32A08_B_C01_CPC_IRQHandler
        ISR_HANDLER INTT32A09_A_CT_IRQHandler 
        ISR_HANDLER INTT32A09_B_C01_CPC_IRQHandler
        ISR_HANDLER INTT32A10_A_CT_IRQHandler 
        ISR_HANDLER INTT32A10_B_C01_CPC_IRQHandler
        ISR_HANDLER INTT32A11_A_CT_IRQHandler 
        ISR_HANDLER INTT32A11_B_C01_CPC_IRQHandler
        ISR_HANDLER INTT32A12_A_CT_IRQHandler 
        ISR_HANDLER INTT32A12_B_C01_CPC_IRQHandler
        ISR_HANDLER INTT32A13_A_CT_IRQHandler 
        ISR_HANDLER INTT32A13_B_C01_CPC_IRQHandler
        ISR_HANDLER INTEMG0_IRQHandler        
        ISR_HANDLER INTOVV0_IRQHandler        
        ISR_HANDLER INTPWM0_IRQHandler        
        ISR_HANDLER INTT0RX_IRQHandler        
        ISR_HANDLER INTT0TX_IRQHandler        
        ISR_HANDLER INTT0ERR_IRQHandler       
        ISR_HANDLER INTT1RX_IRQHandler        
        ISR_HANDLER INTT1TX_IRQHandler        
        ISR_HANDLER INTT1ERR_IRQHandler       
        ISR_HANDLER INTT2RX_IRQHandler        
        ISR_HANDLER INTT2TX_IRQHandler        
        ISR_HANDLER INTT2ERR_IRQHandler       
        ISR_HANDLER INTT3RX_IRQHandler        
        ISR_HANDLER INTT3TX_IRQHandler        
        ISR_HANDLER INTT3ERR_IRQHandler       
        ISR_HANDLER INTT4RX_IRQHandler        
        ISR_HANDLER INTT4TX_IRQHandler        
        ISR_HANDLER INTT4ERR_IRQHandler       
        ISR_HANDLER INTT5RX_IRQHandler        
        ISR_HANDLER INTT5TX_IRQHandler        
        ISR_HANDLER INTT5ERR_IRQHandler       
        ISR_HANDLER INTT6RX_IRQHandler        
        ISR_HANDLER INTT6TX_IRQHandler        
        ISR_HANDLER INTT6ERR_IRQHandler       
        ISR_HANDLER INTT7RX_IRQHandler        
        ISR_HANDLER INTT7TX_IRQHandler        
        ISR_HANDLER INTT7ERR_IRQHandler       
        ISR_HANDLER INTT8RX_IRQHandler        
        ISR_HANDLER INTT8TX_IRQHandler        
        ISR_HANDLER INTT8ERR_IRQHandler       
        ISR_HANDLER INTSMI0_IRQHandler        
        ISR_HANDLER INTUART0RX_IRQHandler     
        ISR_HANDLER INTUART0TX_IRQHandler     
        ISR_HANDLER INTUART0ERR_IRQHandler    
        ISR_HANDLER INTUART1RX_IRQHandler     
        ISR_HANDLER INTUART1TX_IRQHandler     
        ISR_HANDLER INTUART1ERR_IRQHandler    
        ISR_HANDLER INTUART2RX_IRQHandler     
        ISR_HANDLER INTUART2TX_IRQHandler     
        ISR_HANDLER INTUART2ERR_IRQHandler    
        ISR_HANDLER INTUART3RX_IRQHandler     
        ISR_HANDLER INTUART3TX_IRQHandler     
        ISR_HANDLER INTUART3ERR_IRQHandler    
        ISR_HANDLER INTUART4RX_IRQHandler     
        ISR_HANDLER INTUART4TX_IRQHandler     
        ISR_HANDLER INTUART4ERR_IRQHandler    
        ISR_HANDLER INTUART5RX_IRQHandler     
        ISR_HANDLER INTUART5TX_IRQHandler     
        ISR_HANDLER INTUART5ERR_IRQHandler    
        ISR_HANDLER INTFUART0_IRQHandler      
        ISR_HANDLER INTFUART1_IRQHandler      
        ISR_HANDLER INTI2C0NST_IRQHandler     
        ISR_HANDLER INTI2C0ATX_IRQHandler     
        ISR_HANDLER INTI2C0BRX_IRQHandler     
        ISR_HANDLER INTI2C0NA_IRQHandler      
        ISR_HANDLER INTI2C1NST_IRQHandler     
        ISR_HANDLER INTI2C1ATX_IRQHandler     
        ISR_HANDLER INTI2C1BRX_IRQHandler     
        ISR_HANDLER INTI2C1NA_IRQHandler      
        ISR_HANDLER INTI2C2NST_IRQHandler     
        ISR_HANDLER INTI2C2ATX_IRQHandler     
        ISR_HANDLER INTI2C2BRX_IRQHandler     
        ISR_HANDLER INTI2C2NA_IRQHandler      
        ISR_HANDLER INTI2C3NST_IRQHandler     
        ISR_HANDLER INTI2C3ATX_IRQHandler     
        ISR_HANDLER INTI2C3BRX_IRQHandler     
        ISR_HANDLER INTI2C3NA_IRQHandler      
        ISR_HANDLER INTI2C4NST_IRQHandler     
        ISR_HANDLER INTI2C4ATX_IRQHandler     
        ISR_HANDLER INTI2C4BRX_IRQHandler     
        ISR_HANDLER INTI2C4NA_IRQHandler      
        ISR_HANDLER INTADACP0_IRQHandler      
        ISR_HANDLER INTADACP1_IRQHandler      
        ISR_HANDLER INTADATRG_IRQHandler      
        ISR_HANDLER INTADASGL_IRQHandler      
        ISR_HANDLER INTADACNT_IRQHandler      
        ISR_HANDLER INTADAHP_IRQHandler       
        ISR_HANDLER INTFLDRDY_IRQHandler      
        ISR_HANDLER INTFLCRDY_IRQHandler      
        ISR_RESERVED_DUMMY                         
        ISR_RESERVED_DUMMY                         
        ISR_HANDLER INTMDMAABERR_IRQHandler   
        ISR_HANDLER INTMDMAADERR_IRQHandler   
        ISR_HANDLER INTI2S0SI_IRQHandler      
        ISR_HANDLER INTI2S0SIERR_IRQHandler   
        ISR_HANDLER INTI2S0SO_IRQHandler      
        ISR_HANDLER INTI2S0SOERR_IRQHandler   
        ISR_HANDLER INTI2S1SI_IRQHandler      
        ISR_HANDLER INTI2S1SIERR_IRQHandler   
        ISR_HANDLER INTI2S1SO_IRQHandler      
        ISR_HANDLER INTI2S1SOERR_IRQHandler   
        ISR_HANDLER INTFIR0_IRQHandler        
        ISR_HANDLER INTTSSI0RX_IRQHandler     
        ISR_HANDLER INTTSSI0TX_IRQHandler     
        ISR_HANDLER INTTSSI0ERR_IRQHandler    
        ISR_HANDLER INTTSSI1RX_IRQHandler     
        ISR_HANDLER INTTSSI1TX_IRQHandler     
        ISR_HANDLER INTTSSI1ERR_IRQHandler    
        ISR_HANDLER INTEMA_IRQHandler         
        ISR_HANDLER INTEMAMP_IRQHandler       
        ISR_HANDLER INTCAN0RX_IRQHandler      
        ISR_HANDLER INTCAN0TX_IRQHandler      
        ISR_HANDLER INTCAN0GLB_IRQHandler     
        ISR_HANDLER INTCAN1RX_IRQHandler      
        ISR_HANDLER INTCAN1TX_IRQHandler      
        ISR_HANDLER INTCAN1GLB_IRQHandler     
        ISR_HANDLER INTUSBOTGA_IRQHandler     
        ISR_HANDLER INTUSBOTGAWK_IRQHandler   
        ISR_HANDLER INTUSBOTGB_IRQHandler     
        ISR_HANDLER INTUSBOTGBWK_IRQHandler   
        ISR_HANDLER INTT32A14_A_IRQHandler    
        ISR_HANDLER INTT32A14_B_IRQHandler    
        ISR_HANDLER INTT32A14_CT_IRQHandler   
        ISR_HANDLER INTT32A15_A_IRQHandler    
        ISR_HANDLER INTT32A15_B_IRQHandler    
        ISR_HANDLER INTT32A15_C_IRQHandler    
        ISR_HANDLER INTMDMAADISR_IRQHandler   
#endif
        //
        .section .vectors, "ax"
_vectors_end:

#ifdef __VECTORS_IN_RAM
        //
        // Reserve space with the size of the vector table
        // in the designated RAM section.
        //
        .section .vectors_ram, "ax"
        .balign 512
        .global _vectors_ram

_vectors_ram:
        .space _vectors_end - _vectors, 0
#endif

/*********************************************************************
*
*  Dummy handler to be used for reserved interrupt vectors
*  and weak implementation of interrupts.
*
*/
        .section .init.Dummy_Handler, "ax"
        .thumb_func
        .weak Dummy_Handler
        .balign 2
Dummy_Handler:
        1: b 1b   // Endless loop


/*************************** End of file ****************************/
