#!/usr/bin/python3
#/*********************************************************************
#*               (c) SEGGER Microcontroller GmbH & Co. KG             *
#*                        The Embedded Experts                        *
#*                           www.segger.com                           *
#**********************************************************************
#
#-------------------------- END-OF-HEADER -----------------------------
#
#Purpose : Flasher Hub Telnet Terminal Example for Python 3
#Literature:
#  [1]  Python 3 Online Documentation https://docs.python.org/3/library/index.html
#  [2]  Flasher Hub Online Documentation https://wiki.segger.com/Flasher_Hub
#
#Notes:
#  ...
#
#Additional information:
#  ...
#*/
#/*********************************************************************
#*
#*       Version Check
#*
#**********************************************************************
#*/
import sys

if sys.version_info < (3, 6, 5):
  sys.stderr.write("You need python 3.6.5 or later to run this script\n")
  if sys.version_info < (3, 0, 0):
    sys.stderr.write("Warning: using Python 2.x despite explicitly asking for Python 3!\n")
    raw_input("Press <Return> to exit")
  else:
    input("Press <Return> to exit")
  sys.exit(1)
  
import socket
import select
import time

#/*********************************************************************
#*
#*       Defines, configurable
#*
#**********************************************************************
#*/

_MODULE_LIST = [1, 2, 3]

#/*********************************************************************
#*
#*       Defines, fixed
#*
#**********************************************************************
#*/

_TELNET_PORT = 23

#/*********************************************************************
#*
#*       Classes
#*
#**********************************************************************
#*/

#/*********************************************************************
#*
#*       _MSG_BUF
#*
#*  Class description
#*    Terminal message buffer
#*/
class _MSG_BUF:
  def __init__(self):
    self.asMsg       = []
    self.NumParsed   = 0
    self.sPartialMsg = None

#/*********************************************************************
#*
#*       Globals
#*
#**********************************************************************
#*/

_MsgBuf = _MSG_BUF()

#/*********************************************************************
#*
#*       Functions
#*
#**********************************************************************
#*/

#/*********************************************************************
#*
#*       _AddToMsgBuf()
#*
#*  Function description
#*    Adds received data to the terminal message buffer.
#*
#*  Parameters
#*    aBytes      bytes() object of data to add
#*
#*  Return value
#*    Number of complete messages added to buffer
#*/
def _AddToMsgBuf(aBytes):
  global _MsgBuf
  #
  # Convert to string
  #
  sAll = aBytes.decode("utf-8")
  #
  # Split into messages starting with '#' and ending in '\r\n'
  #
  NumMsg = 0
  sMsg = _MsgBuf.sPartialMsg  # Partial message available? => Add to it
  for c in sAll:
    if (c == '#'):  # Start of new message?
      if (sMsg):    # We are inside a message? => Terminate it
        sMsg = ""
      else:
        sMsg = c
    elif (c == '\r'):  # End of message?
      if (sMsg):       # Inside a message?
        NumMsg += 1
        _MsgBuf.asMsg.append(sMsg)
        print(f" < {sMsg}")
        if (_MsgBuf.sPartialMsg):
          _MsgBuf.sPartialMsg = None
        sMsg = None
    else:
      if (sMsg):   # Inside a message?
        sMsg += c
  #
  # Given bytes ended in partial message? => Remember it
  #
  if (sMsg):
    if (_MsgBuf.sPartialMsg):  # A partial message already buffered? => Append to it
      _MsgBuf.sPartialMsg += sMsg
    else:                     # No partial message buffered? => Make this it now
      _MsgBuf.sPartialMsg = sMsg
  return NumMsg

#/*********************************************************************
#*
#*       _RecvData()
#*
#*  Function description
#*    Blocks until data from the terminal was read 
#*
#*  Parameters
#*    hSock      Socket to read data from
#*
#*  Return value
#*    Number of complete messages read from terminal
#*/
def _RecvData(hSock):
  #
  # Wait for socket to become readable
  #
  while(True):
    aTmp = select.select([hSock], [], [], 250)
    aTmp = aTmp[0]
    if hSock in aTmp:
      break
  #
  # Receive whatever data is available
  #
  ab = hSock.recv(4096)
  NumMsg = _AddToMsgBuf(ab)
  return NumMsg

#/*********************************************************************
#*
#*       _WaitForResult()
#*
#*  Function description
#*    Blocks until a #RESULT:[...] message from the terminal was received
#*
#*  Parameters
#*    hSock      Socket of terminal connection
#*
#*  Return value
#*    Result message received from terminal
#*/
def _WaitForResult(hSock):
  global _MsgBuf
  print(f"Waiting for result...")
  sResp = "#RESULT:"
  while(True):
    #
    # Look through messages that were received but not parsed yet
    #
    NumToParse  = len(_MsgBuf.asMsg)
    NumToParse -= _MsgBuf.NumParsed
    FirstIndex  = _MsgBuf.NumParsed
    LastIndex   = FirstIndex + NumToParse
    for i in range (FirstIndex, LastIndex):
      sMsg = _MsgBuf.asMsg[i]
      _MsgBuf.NumParsed += 1
      if (sMsg.startswith(sResp)):  # Message is the expected response? => Done
        return sMsg
    #
    # Response not found in buffered messages? => Receive some more
    #
    while (True):
      NumMsgRcvd = _RecvData(hSock)
      if (NumMsgRcvd):
        break;
      #
      # No new messages received? => Wait for some time before trying again
      #
      time.sleep(0.250)

#/*********************************************************************
#*
#*       _WaitForResp()
#*
#*  Function description
#*    Blocks until the given response from the terminal was received
#*
#*  Parameters
#*    hSock      Socket of terminal connection
#*    sResp      Response to wait for, e.g. "ACK" if response to wait for is "#ACK\r\n"
#*/
def _WaitForResp(hSock, sResp):
  global _MsgBuf
  print(f"Waiting for {sResp}...")
  sResp = "#" + sResp
  while(True):
    #
    # Look through messages that were received but not parsed yet
    #
    NumToParse  = len(_MsgBuf.asMsg)
    NumToParse -= _MsgBuf.NumParsed
    FirstIndex  = _MsgBuf.NumParsed
    LastIndex   = FirstIndex + NumToParse
    Found = 0
    for i in range (FirstIndex, LastIndex):
      sMsg = _MsgBuf.asMsg[i]
      _MsgBuf.NumParsed += 1
      if (sMsg == sResp):  # Message is the expected response? => Done
        Found = 1
        break
    if (Found):  # Response found in buffered messages? => Done
      break
    #
    # Response not found in buffered messages? => Receive some more
    #
    while (True):
      NumMsgRecvd = _RecvData(hSock)
      if (NumMsgRecvd):
        break;
      #
      # No new messages received? => Wait for some time before trying again
      #
      time.sleep(0.250)

#/*********************************************************************
#*
#*       _ParseResult()
#*
#*  Function description
#*    Parses a #RESULT string from the terminal
#*
#*  Parameters
#*    sResult   #RESULT string to parse
#*
#*  Return value
#*       None     Error
#*    != None
#*                [0] Module position
#*                [1] Actual result
#*/
def _ParseResult(sResult):
  #
  # Example result string:
  # #RESULT:4:OK (Total 1s)
  #
  aTmp = sResult.split(':')
  #
  # Example expected list:
  # [0] #RESULT
  # [1] 4
  # [2] OK (Total 1s)
  #
  Module = int(aTmp[1])
  sResult = aTmp[2]
  return Module, sResult

#/*********************************************************************
#*
#*       _SendMsg()
#*
#*  Function description
#*    Sends a message to the terminal.
#*
#*  Parameters
#*    hSock     Socket of terminal connection
#*    sMsg      Message to send without leading '#' or trailing '\r\n', e.g. "AUTO"
#*/
def _SendMsg(hSock, sMsg):
  sDisplay = "#" + sMsg
  sMsg = sDisplay + "\r\n"
  aBytes = sMsg.encode("utf-8")
  hSock.send(aBytes)
  print(f" > {sDisplay}")

#/*********************************************************************
#*
#*       _ExecAUTO()
#*
#*  Function description
#*    Sends the #AUTO command and waits for the responses
#*
#*  Parameters
#*    hSock     Socket of terminal connection
#*    aModules  List of modules to execute command for
#*
#*  Return value
#*      0    OK
#*    < 0    Error
#*/
def _ExecAUTO(hSock, aModules):
  aCopy = aModules.copy()  # Get copy of module list so we do not modify the origin, later on
  #
  # Build command, example:
  #   aModules == [1, 2, 3]
  #   -->
  #   "#AUTO 1,2,3"
  #
  sCmd = "AUTO "
  asModules = []
  for Module in aCopy:
    sModule = str(Module)
    asModules.append(sModule)
  sCmd += ",".join(asModules)
  #
  # Example sequence:
  # > #AUTO 1,2,3
  # < #ACK
  # < #RESULT:1:OK (Total 1s)
  # < #RESULT:3:OK (Total 1s)
  # < #RESULT:2:OK (Total 1s)
  # < #DONE
  #
  _SendMsg(hSock, sCmd)            # Send #AUTO
  _WaitForResp(hSock, "ACK")       # Wait for #ACK
  #
  # Wait until all results arrived
  #
  while (len(aCopy)):
    sResult = _WaitForResult(hSock)  # Wait for #RESULT
    Module, sResult = _ParseResult(sResult)
    aCopy.remove(Module)
  #
  # Wait for final response for command
  #
  _WaitForResp(hSock, "DONE")      # Wait for #DONE

#/*********************************************************************
#*
#*       Entry point for this script
#*
#**********************************************************************
#*/
if __name__ == "__main__": # only executed if this module is executed directly, not if its imported
  #
  # Print header
  #
  print("Flasher Hub Telnet Terminal Example for Python 3")
  print("SEGGER Microcontroller GmbH\twww.segger.com")
  print("Wiki: https://wiki.segger.com/Flasher_Hub")
  print("")
  #
  # Prompt user for info
  #
  sDest = input("Flasher Hub (IP address or hostname): > ")
  sDest = sDest.strip()  # Strip away leading and trailing whitespace from input
  with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as hSock:  # Open socket for terminal connection
    #
    # Connect to terminal via Telnet
    #
    hSock.connect((sDest, _TELNET_PORT))
    #
    # Execute #AUTO command
    #
    _ExecAUTO(hSock, _MODULE_LIST)
  print("Finished.")
  print("")
  input("Press <Return> to exit...")