/*********************************************************************
*                    SEGGER Microcontroller GmbH                     *
*                        The Embedded Experts                        *
**********************************************************************

-------------------------- END-OF-HEADER -----------------------------

File    : main_boot.c
Purpose : Generic bootloader start

*/

#include <stdio.h>
#include <stdlib.h>
#include <stm32f4xx.h>

/*********************************************************************
*
*       Global data
*
**********************************************************************
*/
extern void JumpToApplication(unsigned int);

/*********************************************************************
*
*       Defines, fixed
*
**********************************************************************
*/
#define FLASH_IMAGE_START 0x08004000
#define FLASH_IMAGE_END   0x0807FFFC
#define FLASH_APPLICATION_END (FLASH_IMAGE_END - 0x4)

/*********************************************************************
*
*       Local functions
*
**********************************************************************
*/
/*********************************************************************
*
*       _CalcCRC()
*
*  Function description
*    Calculates the CRC of the downloaded Application image using the
*    hardware CRC module.
*
*  Return value
*     0: OK
*    -1: Error
*/
static int _CalcCRC(void) {
  int i;
  unsigned int NumItems;
  unsigned int* pData;
  unsigned int CRCResultHW;
  unsigned int CRCResult;
  unsigned int OldValue;
  
  i = 0;
  NumItems = (FLASH_APPLICATION_END - FLASH_IMAGE_START) / 4;
  pData = (unsigned int*)FLASH_IMAGE_START;    // points to start of Flash
  CRCResult = *(unsigned int*)FLASH_IMAGE_END; // Saves CRC value calculated by SEGGER Linker
  //
  // Config CRC Module
  //
  RCC->AHB1ENR |= RCC_AHB1ENR_CRCEN;
  CRC->CR |= CRC_CR_RESET;
  //
  // Calculate CRC with ST CRC unit over complete Flash area
  //
  do {
    CRC->DR = __REV(*pData);  // ST algorithm expects words in reversed order
    pData++;
  } while (NumItems--);
  CRCResultHW = CRC->DR;
  printf("Hardware calculated CRC over Flash is: 0x%X \n", CRCResultHW);
  printf("SEGGER Linker calculated CRC over Flash is: 0x%X\n", CRCResult);
  //
  // Compare with Linker result
  //
  if (CRCResult == CRCResultHW) {
    printf("Both CRC check sums are matching!\n");
    return 0;
  } else {
    printf("CRC check sums are not matching. Check parameters.\n");
    return -1;
  }
}

/*********************************************************************
*
*       main()
*
*  Function description
*   Bootloader entry point.
*/
int main(void) {
  int r;
  unsigned int VectorTableAddress;
  unsigned int* p = (unsigned int*)(FLASH_IMAGE_START); // location of boot header
  VectorTableAddress = p[0];
  //
  // Print bootheader content
  //
  printf("Detected Bootheader:\n\n");
  printf("Vector table address: 0x%x\n", p[0]);
  printf("Flash used: %d\n", p[1]);
  printf("Flash start: 0x%x\n", p[2]);
  printf("Flash end: 0x%x\n", p[3]);
  //
  // Do CRC check
  //
  r = _CalcCRC();
  if (r != -1) {
    printf("SUCCESS: CRC check sucessfull, starting application!\n\n");
    JumpToApplication(VectorTableAddress);
  } else {
    printf("ERROR: Invalid, or no image found! Boot aborted.\n");
    while(1);
  }
}

/*************************** End of file ****************************/
