/*********************************************************************
*                     SEGGER Microcontroller GmbH                    *
*                        The Embedded Experts                        *
**********************************************************************
*                                                                    *
*       (c) 1995 - 2023 SEGGER Microcontroller GmbH                  *
*                                                                    *
*       Internet: segger.com  Support: support_embos@segger.com      *
*                                                                    *
**********************************************************************
*                                                                    *
*       embOS * Real time operating system                           *
*                                                                    *
*       Please note:                                                 *
*                                                                    *
*       Knowledge of this file may under no circumstances            *
*       be used to write a similar product or a real-time            *
*       operating system for in-house use.                           *
*                                                                    *
*       Thank you for your fairness !                                *
*                                                                    *
**********************************************************************
*                                                                    *
*       OS version: V5.18.0.0                                        *
*                                                                    *
**********************************************************************

-------------------------- END-OF-HEADER -----------------------------
File    : main.c
Purpose : Showcasing the SystemView heap functionality.
Additional information:
  https://wiki.segger.com/How_to_use_SystemView_Heap_Monitoring
*/

#include "RTOS.h"
#include "SEGGER_SYSVIEW.h"
#include "BSP.h"
#include <stdlib.h>

#define NUM_TASKS               3
#define TASK_DELAY              10
#define TASK_STACK_SIZE         128
#define TASK_MAX_PRIO           100
#define HEAP_SIZE               1024
#define HEAP_CHUNK_SIZE         8     // for all archs lower than 64-bit, otherwise it should be 16
#define DEFAULT_ALLOC_SIZE      128

// Only one of the following should be active based on the project's 'Library Heap' setting:
#define HEAP_LIB_BASIC
//#define HEAP_LIB_REALTIME
//#define HEAP_LIB_MINIMAL



static void             _Task1(void);
static void             _Task2(void);
static void             _Task3(void);
static OS_STACKPTR int  _Stack[NUM_TASKS][TASK_STACK_SIZE];
static OS_TASK          _TCB[NUM_TASKS];
static OS_MAILBOX       _MB;
static void*            _MBBuf;
static void (*_Task[NUM_TASKS])(void)           = {_Task1, _Task2, _Task3};
static const char*      _asTaskName[NUM_TASKS]  = { "Task 0", "Task 1", "Task 2" };

extern unsigned int __heap_start__[];
extern unsigned int __heap_size__[];


void MAIN_SetupSysVInstr  (void);
void* __wrap_malloc   (size_t Size);
void* __wrap_realloc  (void* p, size_t Size);
void  __wrap_free     (void* pMem);


// Prototypes of the original stdlib functions symbols created during the linker wrapping functionality
void* __real_malloc       (size_t size);
void* __real_realloc      (void* p, size_t size);
void  __real_free         (void* p);

/*********************************************************************
*
*       MAIN_SetupSysVInstr
*
*  Function description
*    Specifies the heap to be used with the SystemView
*/
void MAIN_SetupSysVInstr(void) {
  I32 metaDataSize;

#if defined HEAP_LIB_MINIMAL
  metaDataSize = 0;
#elif defined HEAP_LIB_BASIC
  metaDataSize  = sizeof(unsigned int) + sizeof(void*);
#elif defined HEAP_LIB_REALTIME
  metaDataSize = sizeof(unsigned int) + 3 * sizeof(void*);
#endif
  SEGGER_SYSVIEW_HeapDefine(__heap_start__, __heap_start__, (unsigned int)__heap_size__, metaDataSize);
  SEGGER_SYSVIEW_NameResource((U32)__heap_start__, "System Heap");
}

/*********************************************************************
*
*       _PseudoRandomByte
*
*  Function description
*    Returns the pseudo-random value 0-255 based on the tick counter
*
*  Return value
*    The pseudo-random value
*/
static U32 _PseudoRandomByte(void) {
  U32 r;
 
  r = OS_TIME_GetTicks32() & 0x000000ff;
  if (r == 0) {
    r = DEFAULT_ALLOC_SIZE;
  }
  return r;
}

/*********************************************************************
*
*       _GetAllocBlockSize
*
*  Function description
*    Returns the real block size internally reserved for the requested
*    allocation size for the 'Basic' SEGGER heap model
*
*  Parameters
*    ReqSize : requested size
*
*  Return value
*    block size
*/
static U32 _GetAllocBlockSize(U32 ReqSize) {
#if defined HEAP_LIB_MINIMAL
  return (ReqSize + HEAP_CHUNK_SIZE) & (0u - HEAP_CHUNK_SIZE);
#elif defined HEAP_LIB_BASIC || defined HEAP_LIB_REALTIME
  return (ReqSize + sizeof(unsigned) + HEAP_CHUNK_SIZE - 1) & (unsigned)(-HEAP_CHUNK_SIZE);
#endif
}


/*********************************************************************
*
*       __wrap_malloc
*
*  Function description
*    Replaces the stdlib malloc function and dispatches the 
*    SystemView HeapAlloc event.
*
*  Parameters
*    Size : Size of the memory block to be allocated
*
*  Return value
*    Pointer to the allocated memory
*/
void* __wrap_malloc(size_t Size) {
  void* pMem;
  
  pMem = __real_malloc(Size);
  // SystemView needs the real internal heap block size, not the amount requested by the user,
  // hence the call to a helper function _GetAllocBlockSize for the currently selected heap type ('Basic')
  SEGGER_SYSVIEW_HeapAlloc(__heap_start__, pMem, _GetAllocBlockSize(Size));
  return pMem;
}

/*********************************************************************
*
*       __wrap_realloc
*
*  Function description
*    Replaces the stdlib realloc function and dispatches the SystemView HeapFree and HeapAlloc event.
*
*  Parameters
*    pOld : Pointer to the memory to be reallocated
*    Size : Size of the reallocated memory block
*
*  Return value
*    Pointer to the allocated memory
*/
void* __wrap_realloc(void* pOld, size_t Size) {
  void* pNew;

#ifndef HEAP_LIB_MINIMAL
  pNew = __real_realloc(pOld, Size);
  SEGGER_SYSVIEW_HeapFree(__heap_start__, pOld);
  // SystemView needs the real internal heap block size, not the amount requested by the user,
  // hence the call to a helper function _GetAllocBlockSize for the currently selected heap type ('Basic')
  SEGGER_SYSVIEW_HeapAlloc(__heap_start__, pNew, _GetAllocBlockSize(Size));
#else
  return NULL;
#endif
  return pNew;
}

/*********************************************************************
*
*       __wrap_free
*
*  Function description
*    Replaces the stdlib free function and dispatches the SystemView HeapFree event.
*
*  Parameters
*    pMem : Pointer to be released
*/
void __wrap_free(void* pMem) {
  __real_free(pMem);
#ifndef HEAP_LIB_MINIMAL
  SEGGER_SYSVIEW_HeapFree(__heap_start__, pMem);
#endif
}

/*********************************************************************
*
*       Task1
*
*  Function description
*    Constantly allocates and reallocates memory
*/
static void _Task1(void) {
  void* pMem;
  
  while (1) {
    pMem = malloc(_PseudoRandomByte());
    OS_TASK_Delay(TASK_DELAY);
    pMem = realloc(pMem, _PseudoRandomByte());
    OS_TASK_Delay(TASK_DELAY);
    free(pMem);
    OS_TASK_Delay(TASK_DELAY);
  }
}

/*********************************************************************
*
*       Task2
*
*  Function description
*    Constantly allocates some memory and passes it to the 3rd task
*/
static void _Task2(void) {  
  void* pMem;

  while (1) {
    pMem = malloc(_PseudoRandomByte());
    OS_MAILBOX_PutBlocked(&_MB, &pMem);
    OS_TASK_Delay(TASK_DELAY);
  }
}

/*********************************************************************
*
*       Task3
*
*  Function description
*    Waits for the address to received from the Task 2 and then
*    frees the pointer.
*/
static void _Task3(void) {
  void* pMem;

  while (1) {
    pMem = NULL;
    OS_MAILBOX_GetBlocked(&_MB, &pMem);
    if (pMem != NULL) {
      free(pMem);
    }
    OS_TASK_Delay(TASK_DELAY);
  }
}

/*********************************************************************
*
*       main()
*/
int main(void) {
  int i;

  OS_Init();
  OS_InitHW();
  BSP_Init();
  OS_MAILBOX_Create(&_MB, sizeof(void*), 1, &_MBBuf);
  for (i = 0; i < NUM_TASKS; ++i) {
    OS_TASK_CREATE(&_TCB[i], _asTaskName[i], TASK_MAX_PRIO, _Task[i], _Stack[i]);
  }
  while (SEGGER_SYSVIEW_IsStarted() == 0) {
    ;
  }
  OS_Start();   // Start embOS
  return 0;
}

/*************************** End of file ****************************/
