/*********************************************************************
*                     SEGGER Microcontroller GmbH                    *
*                        The Embedded Experts                        *
**********************************************************************
*                                                                    *
*       (c) 2003 - 2022  SEGGER Microcontroller GmbH                 *
*                                                                    *
*       www.segger.com     Support: support@segger.com               *
*                                                                    *
**********************************************************************
-------------------------- END-OF-HEADER -----------------------------

File    : PIDConf.c
Purpose : Pointer Input Device configuration
*/

#include "PIDConf.h"
//#include "GUI.h"
#include "SEGGER.h"
#include "RTOS.h"

#include "MK66F18.h"  // Device specific header file, contains CMSIS defines.

/*********************************************************************
*
*       Defines, configurable
*
**********************************************************************
*/

//
// Timer intervals for Joystick and User Button, as well as the A/D Converter (used with Potentiometer)
// A value of 0 completely disables the functionality.
//
#define BUTTON_TIMER_INTERVAL         ( 50)  // check button state(s) every 50 ms
#define POTENTIOMETER_TIMER_INTERVAL  (100)  // check potentiometer every 100 ms

/*********************************************************************
*
*       Defines, fixed
*
**********************************************************************
*/

//
// Joystick + User button
//
#define BUTTON_STATE_NONE             (0 << 0)
#define BUTTON_STATE_USER_BUTTON      (1 << 0)
#define BUTTON_STATE_JOYSTICK_SELECT  (1 << 1)
#define BUTTON_STATE_JOYSTICK_RIGHT   (1 << 2)
#define BUTTON_STATE_JOYSTICK_DOWN    (1 << 3)
#define BUTTON_STATE_JOYSTICK_UP      (1 << 4)
#define BUTTON_STATE_JOYSTICK_LEFT    (1 << 5)

/*********************************************************************
*
*       Typesdefs
*
**********************************************************************
*/

typedef struct _PORT_INFO {
  int PortPin;
} _PORT_INFO;

/*********************************************************************
*
*       Prototypes
*
**********************************************************************
*/

void ADC0_IRQHandler(void);

/*********************************************************************
*
*       Static data
*
**********************************************************************
*/

#if POTENTIOMETER_TIMER_INTERVAL
static OS_TIMER         _PotentiometerTimer;
#endif

#if BUTTON_TIMER_INTERVAL
static OS_TIMER         _ButtonTimer;
#endif

static const _PORT_INFO _aButtonInfo[] = {
  {14},  // SW_A (Joystick left)
  {15},  // SW_B (Joystick up)
  {16},  // SW_C (Joystick down)
  {17},  // SW_D (Joystick right)
  {18},  // SW_E (Joystick select)
  {19}   // PB   (User button)
};

/*********************************************************************
*
*       Local functions
*
**********************************************************************
*/

/*********************************************************************
*
*       _GetButtonStatus()
*/
#if BUTTON_TIMER_INTERVAL
static U8 _GetButtonStatus(void) {
  U8 ButtonMask;

  ButtonMask = 0;
  //
  // If read as low it is on, otherwise it is off.
  //
  if ((GPIOC_PDIR & (1 << 19)) == 0) {  // User button
    ButtonMask |= BUTTON_STATE_USER_BUTTON;
  }
  if ((GPIOC_PDIR & (1 << 18)) == 0) {  // Joystick select
    ButtonMask |= BUTTON_STATE_JOYSTICK_SELECT;
  }
  if ((GPIOC_PDIR & (1 << 17)) == 0) {  // Joystick right
    ButtonMask |= BUTTON_STATE_JOYSTICK_RIGHT;
  }
  if ((GPIOC_PDIR & (1 << 16)) == 0) {  // Joystick down
    ButtonMask |= BUTTON_STATE_JOYSTICK_DOWN;
  }
  if ((GPIOC_PDIR & (1 << 15)) == 0) {  // Joystick up
    ButtonMask |= BUTTON_STATE_JOYSTICK_UP;
  }
  if ((GPIOC_PDIR & (1 << 14)) == 0) {  // Joystick left
    ButtonMask |= BUTTON_STATE_JOYSTICK_LEFT;
  }
  return ButtonMask;
}
#endif

/*********************************************************************
*
*       _ExecButton()
*/
#if BUTTON_TIMER_INTERVAL
static void _ExecButton(void) {
  U8 Status;

  Status = _GetButtonStatus();  // Evaluate button status
  if (Status & BUTTON_STATE_USER_BUTTON) {
    
  }
  if (Status & BUTTON_STATE_JOYSTICK_SELECT) {
    BSP_ClrLED(0);
    BSP_ClrLED(1);
    OS_Stop();
  }
  if (Status & BUTTON_STATE_JOYSTICK_RIGHT) {
    
  }
  if (Status & BUTTON_STATE_JOYSTICK_DOWN) {
   
  }
  if (Status & BUTTON_STATE_JOYSTICK_UP) {
   
  }
  if (Status & BUTTON_STATE_JOYSTICK_LEFT) {
   
  }
  
  OS_RetriggerTimer(&_ButtonTimer);  // Make timer periodic
}
#endif

/*********************************************************************
*
*       _ExecPotentiometer()
*/
#if POTENTIOMETER_TIMER_INTERVAL
static void _ExecPotentiometer(void) {
  I16 Value;

  ADC0->SC1[0] |= 20;                               // Start conversion
  while ((ADC0->SC1[0] & ADC_SC1_COCO_MASK) == 0);  // Wait for conversion complete
  Value = ADC0->R[0];
  if (Value >= 0) {                                 // Filter underflows
    TPM2->CONTROLS[0].CnV = Value;                  // Conversion completed, accordingly configure ADC duty cycle
  } else {
    TPM2->CONTROLS[0].CnV = 0;
  }
  OS_RetriggerTimer(&_PotentiometerTimer);          // Make timer periodic
}
#endif

/*********************************************************************
*
*       Global functions
*
**********************************************************************
*/

/*********************************************************************
*
*       PID_X_Init()
*/
void SetupJoystick(void) {
  unsigned int i;
  unsigned int PortPin;

  SIM_SOPT2 |= SIM_SOPT2_TPMSRC(1);  // Select TPM clock source
  SIM_SCGC2 |= SIM_SCGC2_TPM2_MASK;  // turn on clock to TPM2
  //
  // Setup Joystick and User button
  //
#if BUTTON_TIMER_INTERVAL
  for (i = 0; i < SEGGER_COUNTOF(_aButtonInfo); i++) {
    PortPin = _aButtonInfo[i].PortPin;
    PORTC->PCR[PortPin] |= PORT_PCR_PS_MASK  // Pull select
                        |  PORT_PCR_PE_MASK  // Pull enable
                        |  PORT_PCR_MUX(1)   // Alternative 1 (GPIO)
                        ;
    PTC->PDDR           |= (0 << PortPin);   // Set port pin to input
  }
  OS_CREATETIMER(&_ButtonTimer, _ExecButton, BUTTON_TIMER_INTERVAL);
#endif
  //
  // Setup A/D Converter (Potentiometer) for LCD
  //
#if POTENTIOMETER_TIMER_INTERVAL
  SIM_SCGC6    |= SIM_SCGC6_ADC0_MASK;  // turn on clock to ADC0
  ADC0->SC2    |= ADC_SC2_REFSEL(0);    // Select voltage reference
  ADC0->SC1[0]  = ADC_SC1_ADCH(23);     // Select input channel 23 (Potentiometer)
  OS_CREATETIMER(&_PotentiometerTimer, _ExecPotentiometer, POTENTIOMETER_TIMER_INTERVAL);
#endif
}

/*************************** End of file ****************************/
