/*********************************************************************
*                    SEGGER Microcontroller GmbH                     *
*                        The Embedded Experts                        *
**********************************************************************
*                                                                    *
*            (c) 2014 - 2019 SEGGER Microcontroller GmbH             *
*                                                                    *
*       www.segger.com     Support: support@segger.com               *
*                                                                    *
**********************************************************************
*                                                                    *
* All rights reserved.                                               *
*                                                                    *
* Redistribution and use in source and binary forms, with or         *
* without modification, are permitted provided that the following    *
* condition is met:                                                  *
*                                                                    *
* o Redistributions of source code must retain the above copyright   *
*   notice, this condition and the following disclaimer.             *
*                                                                    *
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND             *
* CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,        *
* INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF           *
* MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE           *
* DISCLAIMED. IN NO EVENT SHALL SEGGER Microcontroller BE LIABLE FOR *
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR           *
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT  *
* OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;    *
* OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF      *
* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT          *
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE  *
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH   *
* DAMAGE.                                                            *
*                                                                    *
**********************************************************************

-------------------------- END-OF-HEADER -----------------------------

File    : crt0_ARM.s
Purpose : Runtime initialization and standard library-specific code
          for use with the GNU Linker and Standard Library
          onr ARM, Cortex-A, and Cortex-R processors.

Additional information:

  This file is designed for use with the SEGGER Linker and Standard Library.
  It does not require any modification.

  However, it can be configured by the following preprocessor definitions:
    APP_ENTRY_POINT
      Define the application entry point.
      If not defined, the application entry point is: main

    SUPERVISOR_START
      If defined, the application will start up in supervisor mode. If 
      undefined the application will start up in system mode.

    INITIALIZE_STACKS
      If defined, the contents of the stacks will be initialized to a the value 0xCC.

    INITIALIZE_SECONDARY_SECTIONS
      If defined, the .text2, .data2 and .bss2 sections will be initialized.

    INITIALIZE_USER_SECTIONS
      If defined, user-supplied function InitializeUserMemorySections is called
      to initialize memory sections which are not iautomatically nitialized.

    __HEAP_SIZE__
      (Defined by project options)
      Define the size of the system heap.

    SKIP_CALL_CTORS_DTORS
      If defined, constructors and destructors are not called.
      (Note: In some CPU Packages UpdateSystemCoreClock added to the list of constructors.)

    FULL_LIBRARY
      If defined then 
       - argc, argv are setup by the debug_getargs.
       - the exit symbol is defined and executes on return from main.
       - the exit symbol calls destructors, atexit functions and then debug_exit.
      If not defined then
       - argc and argv are not initialized.
       - the exit symbol is defined, executes on return from main and loops.

    ARGSSPACE
      Define the space to reserve for application arguments,
      to be passed by the debugger.
      If not defined, 128 bytes are reserved.
*/
        .syntax unified
        .arm

/*********************************************************************
*
*       Macros
*
**********************************************************************
*/

/*********************************************************************
*
*       FUNC
*
*    Declare a global function
*/
.macro FUNC Name
        .global \Name
        .type   \Name, function
        .weak   \Name
        .arm
        .balign 4
\Name:
.endm

/*********************************************************************
*
*       HELPER
*
*    Declare a global, weak function, 
*    as used for default library helpers and interrupt handlers.
*/
.macro HELPER Name
        .section .text.\Name, "ax", %progbits
        .global \Name
        .type \Name, function
        .weak \Name  
        .arm
        .balign 4
\Name:
.endm

/*********************************************************************
*
*       Defines, defaults
*
**********************************************************************
*/
#ifndef APP_ENTRY_POINT
#define APP_ENTRY_POINT main
#endif

#ifndef ARGSSPACE
#define ARGSSPACE 128
#endif  

/*********************************************************************
*
*       Extern symbols
*
**********************************************************************
*/

        .extern APP_ENTRY_POINT
#ifdef INITIALIZE_USER_SECTIONS
        .extern InitializeUserMemorySections
#endif

/*********************************************************************
*
*       Global functions
*
**********************************************************************
*/
        .section .init, "ax"  // _start, exit, and exit_loop need to stay in the given order. Place them into one section ".init".

/*********************************************************************
*
*       _start()
*
*  Function description
*    Runtime initialization.
*    Called from reset handler after system initialization.
*    Initialize data in RAM, initialize heap (if used), 
*    call constructors, setup arguments for main (if used),
*    and call the application entry (i.e. main).
*/

        .global __start

  FUNC _start
_start:
__start:                        
        mrs R0, cpsr
        bic R0, R0, #0x1F
        //
        // Setup stacks
        //
        orr R1, R0, #0x1B               // Undefined stack
        msr cpsr_cxsf, R1
        ldr SP, =__stack_und_end__
#ifdef __ARM_EABI__
        bic SP, SP, #0x7
#endif  
        orr R1, R0, #0x17               // Abort stack
        msr cpsr_cxsf, R1
        ldr SP, =__stack_abt_end__
#ifdef __ARM_EABI__
        bic SP, SP, #0x7
#endif
        orr R1, R0, #0x12               // IRQ stack
        msr cpsr_cxsf, R1
        ldr SP, =__stack_irq_end__
#ifdef __ARM_EABI__
        bic SP, SP, #0x7
#endif
        orr R1, R0, #0x11               // FIQ stack
        msr cpsr_cxsf, R1
        ldr SP, =__stack_fiq_end__
#ifdef __ARM_EABI__
        bic SP, SP, #0x7
#endif
        orr R1, R0, #0x13               // Supervisor stack
        msr cpsr_cxsf, R1
        ldr SP, =__stack_svc_end__
#ifdef __ARM_EABI__
        bic SP, SP, #0x7
#endif

#ifdef SUPERVISOR_START
        //
        // Start application in supervisor mode
        // Setup user/system mode stack
        //
        ldr R1, =__stack_end__
#ifdef __ARM_EABI__
        bic R1, R1, #0x7
#endif
        mov R2, SP
        stmfd R2!, {R1}
        ldmfd R2, {SP}^
#else
        //
        // Start application in syste mode
        //
        orr R1, R0, #0x1F               // System mode
        msr cpsr_cxsf, R1
        ldr SP, =__stack_end__
#ifdef __ARM_EABI__
        bic SP, SP, #0x7
#endif
#endif

#ifdef INITIALIZE_STACKS
        //
        // Initialize stacks with 0xCC
        //
        mov R2, #0xCC
        ldr R0, =__stack_und_start__
        ldr R1, =__stack_und_end__
        bl memory_set
        ldr R0, =__stack_abt_start__
        ldr R1, =__stack_abt_end__
        bl memory_set
        ldr R0, =__stack_irq_start__
        ldr R1, =__stack_irq_end__
        bl memory_set
        ldr R0, =__stack_fiq_start__
        ldr R1, =__stack_fiq_end__
        bl memory_set
        ldr R0, =__stack_svc_start__
        ldr R1, =__stack_svc_end__
        bl memory_set  
        ldr R0, =__stack_start__
        ldr R1, =__stack_end__
        bl memory_set  
#endif
        //
        // Copy initialization data into RAM (if necessary)
        //
        ldr R0, =__data_load_start__
        ldr R1, =__data_start__
        ldr R2, =__data_end__
        bl memory_copy
        ldr R0, =__text_load_start__
        ldr R1, =__text_start__
        ldr R2, =__text_end__
        bl memory_copy
        ldr R0, =__fast_load_start__
        ldr R1, =__fast_start__
        ldr R2, =__fast_end__
        bl memory_copy
        ldr R0, =__ctors_load_start__
        ldr R1, =__ctors_start__
        ldr R2, =__ctors_end__
        bl memory_copy
        ldr R0, =__dtors_load_start__
        ldr R1, =__dtors_start__
        ldr R2, =__dtors_end__
        bl memory_copy
        ldr R0, =__rodata_load_start__
        ldr R1, =__rodata_start__
        ldr R2, =__rodata_end__
        bl memory_copy
        ldr R0, =__tdata_load_start__
        ldr R1, =__tdata_start__
        ldr R2, =__tdata_end__
        bl memory_copy
#ifdef INITIALIZE_SECONDARY_SECTIONS
        //
        // Copy initialization data of secondary sections into RAM (if necessary)
        //
        ldr R0, =__data2_load_start__
        ldr R1, =__data2_start__
        ldr R2, =__data2_end__
        bl memory_copy
        ldr R0, =__text2_load_start__
        ldr R1, =__text2_start__
        ldr R2, =__text2_end__
        bl memory_copy
        ldr R0, =__rodata2_load_start__
        ldr R1, =__rodata2_start__
        ldr R2, =__rodata2_end__
        bl memory_copy
#endif
        //
        // Zero the bss and tbss
        //
        ldr R0, =__bss_start__
        ldr R1, =__bss_end__
        mov R2, #0
        bl memory_set
        ldr R0, =__tbss_start__
        ldr R1, =__tbss_end__
        mov R2, #0
        bl memory_set
#ifdef INITIALIZE_SECONDARY_SECTIONS
        //
        // Zero the secondary bss
        //
        ldr R0, =__bss2_start__
        ldr R1, =__bss2_end__
        mov R2, #0
        bl memory_set
#endif

#if !defined(__HEAP_SIZE__) || (__HEAP_SIZE__)
        //
        // Initialize the heap for use with the standard library.
        //
        ldr R0, = __heap_start__
        ldr R1, = __heap_end__
        sub R1, R1, R0
        cmp R1, #8
        movge R2, #0
        strge R2, [R0], #+4
        strge R1, [R0]
#endif

#ifdef INITIALIZE_USER_SECTIONS
        //
        // Initialize additional memory sections
        //
        ldr R2, =InitializeUserMemorySections
        mov LR, PC
#ifdef __ARM_ARCH_3__
        mov PC, R2
#else
        bx R2
#endif
#endif

#ifndef SKIP_CALL_CTORS_DTORS
        //
        // Call the constructors
        //
        ldr R0, =__ctors_start__
        ldr R1, =__ctors_end__
ctor_loop:
        cmp R0, R1
        beq ctor_end
        ldr R2, [R0], #+4
        stmfd SP!, {R0-R1}
        mov LR, PC
#ifdef __ARM_ARCH_3__
        mov PC, R2
#else    
        bx R2
#endif
        ldmfd SP!, {R0-R1}
        b ctor_loop
ctor_end:
#endif
#if defined(FULL_LIBRARY) && (ARGSPACE > 0)
        //
        // Setup arguments for main (argc and argv),
        // taken from the debugger.
        //
        mov R0, #ARGSSPACE
        ldr R1, =args
        ldr R2, =debug_getargs
        mov LR, PC
#ifdef __ARM_ARCH_3__
        mov PC, R2
#else    
        bx R2
#endif 
        ldr R1, =args
#endif
        //
        // Call the application entry (i.e. main).
        //
        ldr R2, =APP_ENTRY_POINT
        mov LR, PC
#ifdef __ARM_ARCH_3__
        mov PC, R2
#else    
        bx R2
#endif
        //
        // Fall-through to exit if appliction returns from main.
        //

/*********************************************************************
*
*       exit()
*
*  Function description
*    Application exit.
*    Called explicitly by application, or on return from main.
*    Call destructors and atexit functions of application, 
*    and send application return result to debugger.
*/
        .type   exit, function
        .global exit
        .weak   exit
exit:
#ifdef FULL_LIBRARY  
        //
        // Save application return result
        //
        mov R5, R0
#ifndef SKIP_CALL_CTORS_DTORS
        //
        // Call desctructors
        //
        ldr R0, =__dtors_start__
        ldr R1, =__dtors_end__
dtor_loop:
        cmp R0, R1
        beq dtor_end
        ldr R2, [R0], #+4
        stmfd SP!, {R0-R1}
        mov LR, PC
#ifdef __ARM_ARCH_3__
        mov PC, R2
#else    
        bx R2
#endif
        ldmfd SP!, {R0-R1}
        b dtor_loop
dtor_end:
#endif
        //
        // Call atexit functions
        //
        ldr R2, =_execute_at_exit_fns
        mov LR, PC
#ifdef __ARM_ARCH_3__
        mov PC, R2
#else    
        bx R2
#endif 
        //
        // Send application return result to debugger.
        // The deubgger might close the debug session or halt the application.
        //
        mov R0, R5
        ldr R2, =debug_exit  
        mov LR, PC
#ifdef __ARM_ARCH_3__
        mov PC, R2
#else    
        bx R2
#endif 
        //
        // Fall-through to endless loop if application continues running.
        //
#endif
        //
        // Loop forever after returning from application entry
        //
exit_loop:
        b exit_loop

/*********************************************************************
*
*       Local functions
*
**********************************************************************
*/
/*********************************************************************
*
*       memory_copy()
*
*  Function description
*    Copy data from source (initialization image) to destination.
*/
memory_copy:
        cmp R0, R1
        moveq PC, LR  
        subs R2, R2, R1
        moveq PC, LR
        //
        // If either pointer ot length is not word aligned then byte copy
        //
        orr R3, R0, R1
        orr R3, R3, R2
        tst R3, #0x3
        bne 2f
1:
        //
        // Word copy
        //
        ldr R3, [R0], #+4
        str R3, [R1], #+4
        subs R2, R2, #4
        bne 1b
        mov PC, LR
2:
        //
        // Byte copy
        //
        ldrb R3, [R0], #+1
        strb R3, [R1], #+1
        subs R2, R2, #1
        bne 2b
        mov PC, LR

/*********************************************************************
*
*       memory_set()
*
*  Function description
*    Set data at destination to given value.
*/
memory_set:
        cmp R0, R1
        moveq PC, LR
        strb R2, [R0], #1
        b memory_set

/*********************************************************************
*
*       Helper functions, required by standard library
*
**********************************************************************
*//*********************************************************************
*
*       __aeabi_read_tp()
*/
HELPER __aeabi_read_tp
        ldr R0, =__tbss_start__-8
        bx LR

/*********************************************************************
*
*       abort()
*/
HELPER abort
        b .

/*********************************************************************
*
*       __assert()
*/
HELPER __assert
        b .

/*********************************************************************
*
*       __aeabi_assert()
*/
HELPER __aeabi_assert
        b .

/*********************************************************************
*
*       __sync_synchronize()
*/
HELPER __sync_synchronize
        bx LR
  
/*********************************************************************
*
*       __getchar()
*/
HELPER __getchar
        b debug_getchar

/*********************************************************************
*
*       __putchar()
*/
HELPER __putchar
        b debug_putchar

/*********************************************************************
*
*       __open()
*/
HELPER __open
        b debug_fopen

/*********************************************************************
*
*       __close()
*/
HELPER __close
        b debug_fclose

/*********************************************************************
*
*       __write()
*/
HELPER __write   
        mov R3, R0
        mov R0, R1
        mov R1, #1  
        b debug_fwrite

/*********************************************************************
*
*       __read()
*/
HELPER __read  
        mov R3, R0
        mov R0, R1
        mov R1, #1 
        b debug_fread


/*********************************************************************
*
*       __seek()
*/
HELPER __seek
        push {R4, LR}
        mov R4, R0
        bl debug_fseek
        cmp R0, #0
        bne 1f
        mov R0, R4
        bl debug_ftell
        b 2f
1:
        ldr R0, =-1
2:
        pop {R4, PC}
        bx LR

/*********************************************************************
*
*       Data
*
**********************************************************************
*/
/*********************************************************************
*
*       char __user_locale_name_buffer[];
*/
        .section .bss.__user_locale_name_buffer, "aw", %nobits
        .global __user_locale_name_buffer
        .weak __user_locale_name_buffer
        __user_locale_name_buffer:
        .word 0x0

#if defined(FULL_LIBRARY) && (ARGSPACE > 0)
        //
        // Reserve some space in .bss for arguments for main.
        //
        .bss
args:
        .space ARGSSPACE
#endif
        //
        // Setup attibutes of stack and heap sections so they don't take up unnecessary room in the elf file
        //
        .section .stack, "wa", %nobits
        .section .stack_abt, "wa", %nobits
        .section .stack_irq, "wa", %nobits
        .section .stack_fiq, "wa", %nobits
        .section .stack_svc, "wa", %nobits
        .section .stack_und, "wa", %nobits
        .section .heap, "wa", %nobits

/*************************** End of file ****************************/
