/*********************************************************************
*                    SEGGER Microcontroller GmbH                     *
*                        The Embedded Experts                        *
**********************************************************************
*                                                                    *
*            (c) 2014 - 2019 SEGGER Microcontroller GmbH             *
*                                                                    *
*       www.segger.com     Support: support@segger.com               *
*                                                                    *
**********************************************************************
*                                                                    *
* All rights reserved.                                               *
*                                                                    *
* Redistribution and use in source and binary forms, with or         *
* without modification, are permitted provided that the following    *
* condition is met:                                                  *
*                                                                    *
* o Redistributions of source code must retain the above copyright   *
*   notice, this condition and the following disclaimer.             *
*                                                                    *
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND             *
* CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,        *
* INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF           *
* MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE           *
* DISCLAIMED. IN NO EVENT SHALL SEGGER Microcontroller BE LIABLE FOR *
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR           *
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT  *
* OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;    *
* OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF      *
* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT          *
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE  *
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH   *
* DAMAGE.                                                            *
*                                                                    *
**********************************************************************

-------------------------- END-OF-HEADER -----------------------------

File    : HardFaultHandler_CortexM.S
Purpose : HardFault exception handler to handle software breakpoints
          when no debugger is attached.
*/

        .syntax unified
        .thumb

/*********************************************************************
*
*       Global functions
*
**********************************************************************
*/

/*********************************************************************
*
*      HardFault_Handler()
*
*  Function description
*    HardFault handler implementation to return on software breakpoints.
*
*  Notes
*    (1) Ensure that HardFault_Handler is part of the exception table
*
*  Additional information
*    Assembler code below is equivalent to the following C-code:
*      void HardFaultHandler(unsigned int* pStack) {
*        if ((NVIC_HFSR >> 31) & 1) {    // We got here because of s BKPT instruction? (Happens when for example the application makes printf() semihosting calls but no debugger is connected to the target)
*          NVIC_HFSR = NVIC_HFSR;        // Reset Hard Fault status by writing [31:31] == 1. This bit is set as we checked it to be == 1 right before this line
*          *(pStack + 6u) += 2u;         // PC is located on stack at SP + 24 bytes. Increment PC by 2 to skip break instruction.
*        } else {
*          while (1) {
*          }
*        }
*      }
*    
*/
        .section .text.HardFault_Handler
        .global HardFault_Handler
        .type   HardFault_Handler, function
        .balign 4
        .thumb_func
HardFault_Handler:
        //
        // This version is for Cortex M3, Cortex M4 and Cortex M4F
        // Copy previously selected stack pointer to R0.
        //
        tst    LR, #4           // Check EXC_RETURN in Link register bit 2.
        ite    EQ
        mrseq  R0, MSP          // Stacking was using MSP.
        mrsne  R0, PSP          // Stacking was using PSP.
        //
        // Check if hardfault has been caused by software break.
        // Otherwise stay in HardFault handler.
        //
        ldr    R1, =0xE000ED2C  // NVIC_HFSR
        ldr    R2, [R1]
        lsls   R2, R2, 1        // Shift Bit 31 into carry.
_Loop:  bcc    _Loop            // Carry == 0? Hardfault has been caused by other reason. Stay in endless loop.
        //
        // Reset hardfault status
        //
        str    R2, [R1]         // NVIC_HFSR = NVIC_HFSR
        //
        // Modify return address to return to next instruction after BKPT
        //
        ldr    R1, [R0, #24]    // R0 == Current SP
        adds   R1, #2
        str    R1, [R0, #24]    // *(pStack + 6u) += 2u;
        //
        // Return -1 in case semihosting is not done.
        //
        mov    R0, #0
        sub    R0, R0, #1
        bx     LR
  .end

/*************************** End of file ****************************/
