/*********************************************************************
*                    SEGGER Microcontroller GmbH                     *
*                        The Embedded Experts                        *
**********************************************************************
*                                                                    *
*            (c) 2014 - 2019 SEGGER Microcontroller GmbH             *
*                                                                    *
*       www.segger.com     Support: support@segger.com               *
*                                                                    *
**********************************************************************
*                                                                    *
* All rights reserved.                                               *
*                                                                    *
* Redistribution and use in source and binary forms, with or         *
* without modification, are permitted provided that the following    *
* condition is met:                                                  *
*                                                                    *
* o Redistributions of source code must retain the above copyright   *
*   notice, this condition and the following disclaimer.             *
*                                                                    *
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND             *
* CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,        *
* INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF           *
* MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE           *
* DISCLAIMED. IN NO EVENT SHALL SEGGER Microcontroller BE LIABLE FOR *
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR           *
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT  *
* OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;    *
* OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF      *
* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT          *
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE  *
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH   *
* DAMAGE.                                                            *
*                                                                    *
**********************************************************************

-------------------------- END-OF-HEADER -----------------------------

File    : Startup_CortexM.s
Purpose : Generic startup for Cortex-M processors.

Additional information:

  This file is designed for use with the SEGGER Linker and Standard Library.
  It does not require any modification.

  However, it can be configured by the following preprocessor definitions:
    __OPTIMIZE_SMALL
      Use one exception handler for all not overwritten exceptions.
      If not defined, each exception uses its on endless loop as an
      exception handler. This way it is possible to debug which 
      not implemented interrupt has been triggered.

    STACK_INIT_VAL
      Overwrite the initial stack pointer placed at the start of the
      vector table.
      If not defined, the symbol __stack_end__ is used.

    __VECTORS
      Define a file to be included, which contains the device-specific
      interrupt handlers and vectors.
      File content should look like:
        ISR_HANDLER ExternalISR0
        ISR_HANDLER ExternalISR1
        ISR_HANDLER ExternalISR2
        ISR_HANDLER ExternalISR3

    NO_STACK_INIT
      If defined, the stack pointer is not initialized by reset handler.
      Should not be defined when software calls reset handler.

    NO_SYSTEM_INIT
      If defined, SystemInit is not called by reset handler.
      SystemInit should initialize the system, such as CPU frequency.
      It is called before any memory (RAM) content is set up.

    MEMORY_INIT
      If defined, MemoryInit is called by reset handler.
      MemoryInit should initialize (external) memories to be accessible.
      It is called before any memory (RAM) content is set up.

    __SOFTFP__
      Usually set by compiler.
      If not defined, floating point unit is enabled.
*/

        .syntax unified
        .thumb

/*********************************************************************
*
*       Macros
*
**********************************************************************
*/
/*********************************************************************
*
*       ISR_HANDLER
*
*    Declare a weak interrupt handler as endless loop,
*    and put it at the current index in the vector table.
*/
#if defined(__OPTIMIZE_SMALL)
.macro ISR_HANDLER Name
        .section .vectors, "ax"
        .word \Name
        .weak \Name
        .thumb_set \Name,Dummy_Handler
.endm
#else
.macro ISR_HANDLER Name
        .section .vectors, "ax"
        .word \Name
        .section .init, "ax"
        .thumb_func
        .weak \Name
\Name:
1:      b 1b /* endless loop */
.endm
#endif

/*********************************************************************
*
*       ISR_RESERVED
*
*    Put a NULL vector at the current index in the vector table.
*/
.macro ISR_RESERVED
        .section .vectors, "ax"
        .word 0
.endm

/*********************************************************************
*
*       FUNC
*
*    Declare a global function
*/
.macro FUNC Name
        .global \Name
        .balign 2
        .thumb_func
\Name:
.endm

/*********************************************************************
*
*       Defines, defaults
*
**********************************************************************
*/
#ifndef STACK_INIT_VAL
#define STACK_INIT_VAL __stack_end__
#endif

/*********************************************************************
*
*       (Assembler) Defines
*
**********************************************************************
*/
        .equ reset_handler, Reset_Handler

/*********************************************************************
*
*       Global symbols
*
**********************************************************************
*/
        .global reset_handler
        .global Reset_Handler
        .global _vectors

/*********************************************************************
*
*       Vector table and default interupt handlers
*
**********************************************************************
*/
        .section .vectors, "ax"
        .balign 2
        //
        // Start of vector table
        //
_vectors:
        //
        // Cortex-M vectors
        //
        .word STACK_INIT_VAL
        .word Reset_Handler
        ISR_HANDLER NMI_Handler
        ISR_HANDLER HardFault_Handler
        ISR_HANDLER MemManage_Handler 
        ISR_HANDLER BusFault_Handler
        ISR_HANDLER UsageFault_Handler
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
        ISR_HANDLER SVC_Handler
        ISR_HANDLER DebugMon_Handler
        ISR_RESERVED
        ISR_HANDLER PendSV_Handler
        ISR_HANDLER SysTick_Handler
        //
        // External vectors
        //
#ifdef __VECTORS
        #include __VECTORS  // Include vectors from file
#else  
        //
        // Add target's interrupts here.
        // Example:
        // ISR_HANDLER ExternalISR0
        // ISR_HANDLER ExternalISR1
        // ISR_HANDLER ExternalISR2
        // ISR_HANDLER ExternalISR3
        //
#endif
        //
        // End of vector table
        //
        .section .vectors, "ax"
_vectors_end:

/*********************************************************************
*
*       Global functions
*
**********************************************************************
*/
        .section .init, "ax"


/*********************************************************************
*
*       Dummy_Handler()
*
*  Function description
*    Dummy exception handler for smallest code.
*    All not overwritten exception vectors end up in this endless loop.
*/
#if defined(__OPTIMIZE_SMALL)
        .thumb_func
Dummy_Handler:
        b .
#endif

/*********************************************************************
*
*       Reset_Handler()
*
*  Function description
*    Interrupt handler, called on reset.
*    Call system initialization and memory initialization,
*    enable floating point unit if required,
*    and call runtime init.
*/
  FUNC Reset_Handler
#ifndef NO_STACK_INIT
        //
        // Set stack pointer.
        // Necessary when Reset_Handler is not called by system reset.
        //
        ldr R0, =STACK_INIT_VAL
        bic R0, #0x7
        mov SP, R0
#endif
#ifndef NO_SYSTEM_INIT
        //
        // Call system initialization
        //
        ldr R0, =SystemInit
        blx R0
#endif

#ifdef MEMORY_INIT
        //
        // Call memory initialization
        //
        ldr R0, =MemoryInit
        blx R0
#endif

#if !defined(__SOFTFP__)
        //
        // Enable floating point unit (CP11 and CP10)
        // CPACR |= (0xf<<20)
        //
        movw R0, 0xED88
        movt R0, 0xE000
        ldr R1, [r0]
        orrs R1, R1, #(0xf << 20)
        str R1, [R0]
        dsb
        isb
#endif
        //
        // Call runtime init.
        //
        b _start

/*********************************************************************
*
*       SystemInit()
*
*  Function description
*    Weak dummy implementation for SystemInit. Simply return.
*/
#ifndef __NO_SYSTEM_INIT
        .thumb_func
        .weak SystemInit
SystemInit:
        bx LR
#endif

/*************************** End of file ****************************/
