/*********************************************************************
*                    SEGGER Microcontroller GmbH                     *
*                        The Embedded Experts                        *
**********************************************************************
*                                                                    *
*            (c) 2014 - 2020 SEGGER Microcontroller GmbH             *
*                                                                    *
*       www.segger.com     Support: support@segger.com               *
*                                                                    *
**********************************************************************
*                                                                    *
* All rights reserved.                                               *
*                                                                    *
* Redistribution and use in source and binary forms, with or         *
* without modification, are permitted provided that the following    *
* condition is met:                                                  *
*                                                                    *
* - Redistributions of source code must retain the above copyright   *
*   notice, this condition and the following disclaimer.             *
*                                                                    *
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND             *
* CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,        *
* INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF           *
* MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE           *
* DISCLAIMED. IN NO EVENT SHALL SEGGER Microcontroller BE LIABLE FOR *
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR           *
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT  *
* OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;    *
* OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF      *
* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT          *
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE  *
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH   *
* DAMAGE.                                                            *
*                                                                    *
**********************************************************************

-------------------------- END-OF-HEADER -----------------------------

File      : TMPM4KNA_Vectors.s
Purpose   : Exception and interrupt vectors for TMPM4KNA devices.

Additional information:
  Preprocessor Definitions
    __NO_EXTERNAL_INTERRUPTS
      If defined,
        the vector table will contain only the internal exceptions
        and interrupts.
    __VECTORS_IN_RAM
      If defined,
        an area of RAM, large enough to store the vector table,
        will be reserved.

    __OPTIMIZATION_SMALL
      If defined,
        all weak definitions of interrupt handlers will share the
        same implementation.
      If not defined,
        all weak definitions of interrupt handlers will be defined
        with their own implementation.
*/
        .syntax unified

/*********************************************************************
*
*       Macros
*
**********************************************************************
*/

//
// Directly place a vector (word) in the vector table
//
.macro VECTOR Name=
        .section .vectors, "ax"
        .code 16
        .word \Name
.endm

//
// Declare an exception handler with a weak definition
//
.macro EXC_HANDLER Name=
        //
        // Insert vector in vector table
        //
        .section .vectors, "ax"
        .word \Name
        //
        // Insert dummy handler in init section
        //
        .section .init.\Name, "ax"
        .thumb_func
        .weak \Name
        .balign 2
\Name:
        1: b 1b   // Endless loop
.endm

//
// Declare an interrupt handler with a weak definition
//
.macro ISR_HANDLER Name=
        //
        // Insert vector in vector table
        //
        .section .vectors, "ax"
        .word \Name
        //
        // Insert dummy handler in init section
        //
#if defined(__OPTIMIZATION_SMALL)
        .section .init, "ax"
        .weak \Name
        .thumb_set \Name,Dummy_Handler
#else
        .section .init.\Name, "ax"
        .thumb_func
        .weak \Name
        .balign 2
\Name:
        1: b 1b   // Endless loop
#endif
.endm

//
// Place a reserved vector in vector table
//
.macro ISR_RESERVED
        .section .vectors, "ax"
        .word 0
.endm

//
// Place a reserved vector in vector table
//
.macro ISR_RESERVED_DUMMY
        .section .vectors, "ax"
        .word Dummy_Handler
.endm

/*********************************************************************
*
*       Externals
*
**********************************************************************
*/
        .extern __stack_end__
        .extern Reset_Handler
        .extern HardFault_Handler

/*********************************************************************
*
*       Global functions
*
**********************************************************************
*/

/*********************************************************************
*
*  Setup of the vector table and weak definition of interrupt handlers
*
*/
        .section .vectors, "ax"
        .code 16
        .balign 512
        .global _vectors
_vectors:
        //
        // Internal exceptions and interrupts
        //
        VECTOR __stack_end__
        VECTOR Reset_Handler
        EXC_HANDLER NMI_Handler
        VECTOR HardFault_Handler
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
        EXC_HANDLER SVC_Handler
        ISR_RESERVED
        ISR_RESERVED
        EXC_HANDLER PendSV_Handler
        EXC_HANDLER SysTick_Handler
        //
        // External interrupts
        //
#ifndef __NO_EXTERNAL_INTERRUPTS
        ISR_HANDLER INT00_IRQHandler          
        ISR_HANDLER INT01_IRQHandler          
        ISR_HANDLER INT02_IRQHandler          
        ISR_HANDLER INT03_IRQHandler          
        ISR_HANDLER INT04_IRQHandler          
        ISR_HANDLER INT05_IRQHandler          
        ISR_HANDLER INT06_IRQHandler          
        ISR_HANDLER INT07_IRQHandler          
        ISR_HANDLER INT08_IRQHandler          
        ISR_HANDLER INT09_IRQHandler          
        ISR_HANDLER INT10_IRQHandler          
        ISR_HANDLER INT11_IRQHandler          
        ISR_HANDLER INT12_IRQHandler          
        ISR_HANDLER INT13_IRQHandler          
        ISR_HANDLER INT14_IRQHandler          
        ISR_HANDLER INT15_IRQHandler          
        ISR_HANDLER INT16_IRQHandler          
        ISR_HANDLER INT17_IRQHandler          
        ISR_HANDLER INT18_IRQHandler          
        ISR_RESERVED_DUMMY                         
        ISR_RESERVED_DUMMY                         
        ISR_HANDLER INT21_IRQHandler          
        ISR_HANDLER INTVCN0_IRQHandler        
        ISR_HANDLER INTVCT0_IRQHandler        
        ISR_HANDLER INTEMG0_IRQHandler        
        ISR_HANDLER INTEMG1_IRQHandler        
        ISR_HANDLER INTEMG2_IRQHandler        
        ISR_HANDLER INTOVV0_IRQHandler        
        ISR_HANDLER INTOVV1_IRQHandler        
        ISR_HANDLER INTOVV2_IRQHandler        
        ISR_HANDLER INTPWM0_IRQHandler        
        ISR_HANDLER INTPWM1_IRQHandler        
        ISR_HANDLER INTPWM2_IRQHandler        
        ISR_HANDLER INTENC00_IRQHandler       
        ISR_HANDLER INTENC01_IRQHandler       
        ISR_HANDLER INTENC10_IRQHandler       
        ISR_HANDLER INTENC11_IRQHandler       
        ISR_HANDLER INTENC20_IRQHandler       
        ISR_HANDLER INTENC21_IRQHandler       
        ISR_HANDLER INTADAPDA_IRQHandler      
        ISR_HANDLER INTADAPDB_IRQHandler      
        ISR_HANDLER INTADACP0_IRQHandler      
        ISR_HANDLER INTADACP1_IRQHandler      
        ISR_HANDLER INTADATRG_IRQHandler      
        ISR_HANDLER INTADASGL_IRQHandler      
        ISR_HANDLER INTADACNT_IRQHandler      
        ISR_HANDLER INTADBPDA_IRQHandler      
        ISR_HANDLER INTADBPDB_IRQHandler      
        ISR_HANDLER INTADBCP0_IRQHandler      
        ISR_HANDLER INTADBCP1_IRQHandler      
        ISR_HANDLER INTADBTRG_IRQHandler      
        ISR_HANDLER INTADBSGL_IRQHandler      
        ISR_HANDLER INTADBCNT_IRQHandler      
        ISR_HANDLER INTADCPDA_IRQHandler      
        ISR_HANDLER INTADCPDB_IRQHandler      
        ISR_HANDLER INTADCCP0_IRQHandler      
        ISR_HANDLER INTADCCP1_IRQHandler      
        ISR_HANDLER INTADCTRG_IRQHandler      
        ISR_HANDLER INTADCSGL_IRQHandler      
        ISR_HANDLER INTADCCNT_IRQHandler      
        ISR_HANDLER INTSC0RX_IRQHandler       
        ISR_HANDLER INTSC0TX_IRQHandler       
        ISR_HANDLER INTSC0ERR_IRQHandler      
        ISR_HANDLER INTSC1RX_IRQHandler       
        ISR_HANDLER INTSC1TX_IRQHandler       
        ISR_HANDLER INTSC1ERR_IRQHandler      
        ISR_HANDLER INTSC2RX_IRQHandler       
        ISR_HANDLER INTSC2TX_IRQHandler       
        ISR_HANDLER INTSC2ERR_IRQHandler      
        ISR_HANDLER INTSC3RX_IRQHandler       
        ISR_HANDLER INTSC3TX_IRQHandler       
        ISR_HANDLER INTSC3ERR_IRQHandler      
        ISR_HANDLER INTI2C0NST_IRQHandler     
        ISR_HANDLER INTI2C0ATX_IRQHandler     
        ISR_HANDLER INTI2C0BRX_IRQHandler     
        ISR_HANDLER INTI2C0NA_IRQHandler      
        ISR_HANDLER INTI2C1NST_IRQHandler     
        ISR_HANDLER INTI2C1ATX_IRQHandler     
        ISR_HANDLER INTI2C1BRX_IRQHandler     
        ISR_HANDLER INTI2C1NA_IRQHandler      
        ISR_RESERVED_DUMMY
        ISR_RESERVED_DUMMY
        ISR_RESERVED_DUMMY
        ISR_HANDLER INTT32A00AC_IRQHandler    
        ISR_HANDLER INTT32A00ACCAP0_IRQHandler
        ISR_HANDLER INTT32A00ACCAP1_IRQHandler
        ISR_HANDLER INTT32A00B_IRQHandler     
        ISR_HANDLER INTT32A00BCAP0_IRQHandler 
        ISR_HANDLER INTT32A00BCAP1_IRQHandler 
        ISR_HANDLER INTT32A01AC_IRQHandler    
        ISR_HANDLER INTT32A01ACCAP0_IRQHandler
        ISR_HANDLER INTT32A01ACCAP1_IRQHandler
        ISR_HANDLER INTT32A01B_IRQHandler     
        ISR_HANDLER INTT32A01BCAP0_IRQHandler 
        ISR_HANDLER INTT32A01BCAP1_IRQHandler 
        ISR_HANDLER INTT32A02AC_IRQHandler    
        ISR_HANDLER INTT32A02ACCAP0_IRQHandler
        ISR_HANDLER INTT32A02ACCAP1_IRQHandler
        ISR_HANDLER INTT32A02B_IRQHandler     
        ISR_HANDLER INTT32A02BCAP0_IRQHandler 
        ISR_HANDLER INTT32A02BCAP1_IRQHandler 
        ISR_HANDLER INTT32A03AC_IRQHandler    
        ISR_HANDLER INTT32A03ACCAP0_IRQHandler
        ISR_HANDLER INTT32A03ACCAP1_IRQHandler
        ISR_HANDLER INTT32A03B_IRQHandler     
        ISR_HANDLER INTT32A03BCAP0_IRQHandler 
        ISR_HANDLER INTT32A03BCAP1_IRQHandler 
        ISR_HANDLER INTT32A04AC_IRQHandler    
        ISR_HANDLER INTT32A04ACCAP0_IRQHandler
        ISR_HANDLER INTT32A04ACCAP1_IRQHandler
        ISR_HANDLER INTT32A04B_IRQHandler     
        ISR_HANDLER INTT32A04BCAP0_IRQHandler 
        ISR_HANDLER INTT32A04BCAP1_IRQHandler 
        ISR_HANDLER INTT32A05AC_IRQHandler    
        ISR_HANDLER INTT32A05ACCAP0_IRQHandler
        ISR_HANDLER INTT32A05ACCAP1_IRQHandler
        ISR_HANDLER INTT32A05B_IRQHandler     
        ISR_HANDLER INTT32A05BCAP0_IRQHandler 
        ISR_HANDLER INTT32A05BCAP1_IRQHandler 
        ISR_HANDLER INTPARI0_IRQHandler       
        ISR_HANDLER INTPARI1_IRQHandler       
        ISR_HANDLER INTDMAATC_IRQHandler      
        ISR_HANDLER INTDMAAERR_IRQHandler     
        ISR_HANDLER INTFLCRDY_IRQHandler      
        ISR_HANDLER INTFLDRDY_IRQHandler      
#endif
        //
        .section .vectors, "ax"
_vectors_end:

#ifdef __VECTORS_IN_RAM
        //
        // Reserve space with the size of the vector table
        // in the designated RAM section.
        //
        .section .vectors_ram, "ax"
        .balign 512
        .global _vectors_ram

_vectors_ram:
        .space _vectors_end - _vectors, 0
#endif

/*********************************************************************
*
*  Dummy handler to be used for reserved interrupt vectors
*  and weak implementation of interrupts.
*
*/
        .section .init.Dummy_Handler, "ax"
        .thumb_func
        .weak Dummy_Handler
        .balign 2
Dummy_Handler:
        1: b 1b   // Endless loop


/*************************** End of file ****************************/
