/*********************************************************************
*                    SEGGER Microcontroller GmbH                     *
*                        The Embedded Experts                        *
**********************************************************************
*                                                                    *
*            (c) 2014 - 2024 SEGGER Microcontroller GmbH             *
*                                                                    *
*       www.segger.com     Support: support@segger.com               *
*                                                                    *
**********************************************************************
*                                                                    *
* All rights reserved.                                               *
*                                                                    *
* Redistribution and use in source and binary forms, with or         *
* without modification, are permitted provided that the following    *
* condition is met:                                                  *
*                                                                    *
* - Redistributions of source code must retain the above copyright   *
*   notice, this condition and the following disclaimer.             *
*                                                                    *
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND             *
* CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,        *
* INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF           *
* MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE           *
* DISCLAIMED. IN NO EVENT SHALL SEGGER Microcontroller BE LIABLE FOR *
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR           *
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT  *
* OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;    *
* OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF      *
* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT          *
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE  *
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH   *
* DAMAGE.                                                            *
*                                                                    *
**********************************************************************

-------------------------- END-OF-HEADER -----------------------------

File      : apollo4b_Vectors.s
Purpose   : Exception and interrupt vectors for apollo4b devices.

Additional information:
  Preprocessor Definitions
    __NO_EXTERNAL_INTERRUPTS
      If defined,
        the vector table will contain only the internal exceptions
        and interrupts.
    __VECTORS_IN_RAM
      If defined,
        an area of RAM, large enough to store the vector table,
        will be reserved.

    __OPTIMIZATION_SMALL
      If defined,
        all weak definitions of interrupt handlers will share the
        same implementation.
      If not defined,
        all weak definitions of interrupt handlers will be defined
        with their own implementation.
*/
        .syntax unified

/*********************************************************************
*
*       Macros
*
**********************************************************************
*/

//
// Directly place a vector (word) in the vector table
//
.macro VECTOR Name=
        .section .vectors, "ax"
        .code 16
        .word \Name
.endm

//
// Declare an exception handler with a weak definition
//
.macro EXC_HANDLER Name=
        //
        // Insert vector in vector table
        //
        .section .vectors, "ax"
        .word \Name
        //
        // Insert dummy handler in init section
        //
        .section .init.\Name, "ax"
        .thumb_func
        .weak \Name
        .balign 2
\Name:
        1: b 1b   // Endless loop
.endm

//
// Declare an interrupt handler with a weak definition
//
.macro ISR_HANDLER Name=
        //
        // Insert vector in vector table
        //
        .section .vectors, "ax"
        .word \Name
        //
        // Insert dummy handler in init section
        //
#if defined(__OPTIMIZATION_SMALL)
        .section .init, "ax"
        .weak \Name
        .thumb_set \Name,Dummy_Handler
#else
        .section .init.\Name, "ax"
        .thumb_func
        .weak \Name
        .balign 2
\Name:
        1: b 1b   // Endless loop
#endif
.endm

//
// Place a reserved vector in vector table
//
.macro ISR_RESERVED
        .section .vectors, "ax"
        .word 0
.endm

//
// Place a reserved vector in vector table
//
.macro ISR_RESERVED_DUMMY
        .section .vectors, "ax"
        .word Dummy_Handler
.endm

/*********************************************************************
*
*       Externals
*
**********************************************************************
*/
        .extern __stack_end__
        .extern Reset_Handler
        .extern HardFault_Handler

/*********************************************************************
*
*       Global functions
*
**********************************************************************
*/

/*********************************************************************
*
*  Setup of the vector table and weak definition of interrupt handlers
*
*/
        .section .vectors, "ax"
        .code 16
        .balign 512
        .global _vectors
_vectors:
        //
        // Internal exceptions and interrupts
        //
        VECTOR __stack_end__
        VECTOR Reset_Handler
        EXC_HANDLER NMI_Handler
        VECTOR HardFault_Handler
#ifdef __ARM_ARCH_6M__
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
#else
        EXC_HANDLER MemManage_Handler
        EXC_HANDLER BusFault_Handler
        EXC_HANDLER UsageFault_Handler
#endif
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
        EXC_HANDLER SVC_Handler
#ifdef __ARM_ARCH_6M__
        ISR_RESERVED
#else
        EXC_HANDLER DebugMon_Handler
#endif
        ISR_RESERVED
        EXC_HANDLER PendSV_Handler
        EXC_HANDLER SysTick_Handler
        //
        // External interrupts
        //
#ifndef __NO_EXTERNAL_INTERRUPTS
        ISR_HANDLER am_brownout_IRQHandler
        ISR_HANDLER am_watchdog_IRQHandler
        ISR_HANDLER am_rtc_IRQHandler
        ISR_HANDLER am_vcomp_IRQHandler
        ISR_HANDLER am_ioslave_ios_IRQHandler
        ISR_HANDLER am_ioslave_acc_IRQHandler
        ISR_HANDLER am_iomaster0_IRQHandler
        ISR_HANDLER am_iomaster1_IRQHandler
        ISR_HANDLER am_iomaster2_IRQHandler
        ISR_HANDLER am_iomaster3_IRQHandler
        ISR_HANDLER am_iomaster4_IRQHandler
        ISR_HANDLER am_iomaster5_IRQHandler
        ISR_HANDLER am_iomaster6_IRQHandler
        ISR_HANDLER am_iomaster7_IRQHandler
        ISR_HANDLER am_ctimer_IRQHandler
        ISR_HANDLER am_uart_IRQHandler
        ISR_HANDLER am_uart1_IRQHandler
        ISR_HANDLER am_uart2_IRQHandler
        ISR_HANDLER am_uart3_IRQHandler
        ISR_HANDLER am_adc_IRQHandler
        ISR_HANDLER am_mspi0_IRQHandler
        ISR_HANDLER am_mspi1_IRQHandler
        ISR_HANDLER am_mspi2_IRQHandler
        ISR_HANDLER am_clkgen_IRQHandler
        ISR_HANDLER am_cryptosec_IRQHandler
        ISR_RESERVED
        ISR_HANDLER am_sdio_IRQHandler
        ISR_HANDLER am_usb_IRQHandler
        ISR_HANDLER am_gpu_IRQHandler
        ISR_HANDLER am_disp_IRQHandler
        ISR_HANDLER am_dsi_IRQHandler
        ISR_RESERVED
        ISR_HANDLER am_stimer_cmpr0_IRQHandler
        ISR_HANDLER am_stimer_cmpr1_IRQHandler
        ISR_HANDLER am_stimer_cmpr2_IRQHandler
        ISR_HANDLER am_stimer_cmpr3_IRQHandler
        ISR_HANDLER am_stimer_cmpr4_IRQHandler
        ISR_HANDLER am_stimer_cmpr5_IRQHandler
        ISR_HANDLER am_stimer_cmpr6_IRQHandler
        ISR_HANDLER am_stimer_cmpr7_IRQHandler
        ISR_HANDLER am_stimerof_IRQHandler
        ISR_RESERVED
        ISR_HANDLER am_audadc0_IRQHandler
        ISR_RESERVED
        ISR_HANDLER am_dspi2s0_IRQHandler
        ISR_HANDLER am_dspi2s1_IRQHandler
        ISR_HANDLER am_dspi2s2_IRQHandler
        ISR_HANDLER am_dspi2s3_IRQHandler
        ISR_HANDLER am_pdm0_IRQHandler
        ISR_HANDLER am_pdm1_IRQHandler
        ISR_HANDLER am_pdm2_IRQHandler
        ISR_HANDLER am_pdm3_IRQHandler
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
        ISR_HANDLER am_gpio0_001f_IRQHandler
        ISR_HANDLER am_gpio0_203f_IRQHandler
        ISR_HANDLER am_gpio0_405f_IRQHandler
        ISR_HANDLER am_gpio0_607f_IRQHandler
        ISR_HANDLER am_gpio1_001f_IRQHandler
        ISR_HANDLER am_gpio1_203f_IRQHandler
        ISR_HANDLER am_gpio1_405f_IRQHandler
        ISR_HANDLER am_gpio1_607f_IRQHandler
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
        ISR_HANDLER am_timer00_IRQHandler
        ISR_HANDLER am_timer01_IRQHandler
        ISR_HANDLER am_timer02_IRQHandler
        ISR_HANDLER am_timer03_IRQHandler
        ISR_HANDLER am_timer04_IRQHandler
        ISR_HANDLER am_timer05_IRQHandler
        ISR_HANDLER am_timer06_IRQHandler
        ISR_HANDLER am_timer07_IRQHandler
        ISR_HANDLER am_timer08_IRQHandler
        ISR_HANDLER am_timer09_IRQHandler
        ISR_HANDLER am_timer10_IRQHandler
        ISR_HANDLER am_timer11_IRQHandler
        ISR_HANDLER am_timer12_IRQHandler
        ISR_HANDLER am_timer13_IRQHandler
        ISR_HANDLER am_timer14_IRQHandler
        ISR_HANDLER am_timer15_IRQHandler
        ISR_HANDLER am_cachecpu_IRQHandler
#endif
        //
        .section .vectors, "ax"
_vectors_end:

#ifdef __VECTORS_IN_RAM
        //
        // Reserve space with the size of the vector table
        // in the designated RAM section.
        //
        .section .vectors_ram, "ax"
        .balign 512
        .global _vectors_ram

_vectors_ram:
        .space _vectors_end - _vectors, 0
#endif

/*********************************************************************
*
*  Dummy handler to be used for reserved interrupt vectors
*  and weak implementation of interrupts.
*
*/
        .section .init.Dummy_Handler, "ax"
        .thumb_func
        .weak Dummy_Handler
        .balign 2
Dummy_Handler:
        1: b 1b   // Endless loop


/*************************** End of file ****************************/
