/*********************************************************************
*                    SEGGER Microcontroller GmbH                     *
*                        The Embedded Experts                        *
**********************************************************************
*                                                                    *
*            (c) 2014 - 2020 SEGGER Microcontroller GmbH             *
*                                                                    *
*       www.segger.com     Support: support@segger.com               *
*                                                                    *
**********************************************************************
*                                                                    *
* All rights reserved.                                               *
*                                                                    *
* Redistribution and use in source and binary forms, with or         *
* without modification, are permitted provided that the following    *
* condition is met:                                                  *
*                                                                    *
* - Redistributions of source code must retain the above copyright   *
*   notice, this condition and the following disclaimer.             *
*                                                                    *
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND             *
* CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,        *
* INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF           *
* MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE           *
* DISCLAIMED. IN NO EVENT SHALL SEGGER Microcontroller BE LIABLE FOR *
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR           *
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT  *
* OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;    *
* OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF      *
* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT          *
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE  *
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH   *
* DAMAGE.                                                            *
*                                                                    *
**********************************************************************

-------------------------- END-OF-HEADER -----------------------------

File      : M3HQA_Vectors.s
Purpose   : Exception and interrupt vectors for M3HQA devices.

Additional information:
  Preprocessor Definitions
    __NO_EXTERNAL_INTERRUPTS
      If defined,
        the vector table will contain only the internal exceptions
        and interrupts.
    __VECTORS_IN_RAM
      If defined,
        an area of RAM, large enough to store the vector table,
        will be reserved.

    __OPTIMIZATION_SMALL
      If defined,
        all weak definitions of interrupt handlers will share the
        same implementation.
      If not defined,
        all weak definitions of interrupt handlers will be defined
        with their own implementation.
*/
        .syntax unified

/*********************************************************************
*
*       Macros
*
**********************************************************************
*/

//
// Directly place a vector (word) in the vector table
//
.macro VECTOR Name=
        .section .vectors, "ax"
        .code 16
        .word \Name
.endm

//
// Declare an exception handler with a weak definition
//
.macro EXC_HANDLER Name=
        //
        // Insert vector in vector table
        //
        .section .vectors, "ax"
        .word \Name
        //
        // Insert dummy handler in init section
        //
        .section .init.\Name, "ax"
        .thumb_func
        .weak \Name
        .balign 2
\Name:
        1: b 1b   // Endless loop
.endm

//
// Declare an interrupt handler with a weak definition
//
.macro ISR_HANDLER Name=
        //
        // Insert vector in vector table
        //
        .section .vectors, "ax"
        .word \Name
        //
        // Insert dummy handler in init section
        //
#if defined(__OPTIMIZATION_SMALL)
        .section .init, "ax"
        .weak \Name
        .thumb_set \Name,Dummy_Handler
#else
        .section .init.\Name, "ax"
        .thumb_func
        .weak \Name
        .balign 2
\Name:
        1: b 1b   // Endless loop
#endif
.endm

//
// Place a reserved vector in vector table
//
.macro ISR_RESERVED
        .section .vectors, "ax"
        .word 0
.endm

//
// Place a reserved vector in vector table
//
.macro ISR_RESERVED_DUMMY
        .section .vectors, "ax"
        .word Dummy_Handler
.endm

/*********************************************************************
*
*       Externals
*
**********************************************************************
*/
        .extern __stack_end__
        .extern Reset_Handler
        .extern HardFault_Handler

/*********************************************************************
*
*       Global functions
*
**********************************************************************
*/

/*********************************************************************
*
*  Setup of the vector table and weak definition of interrupt handlers
*
*/
        .section .vectors, "ax"
        .code 16
        .balign 512
        .global _vectors
_vectors:
        //
        // Internal exceptions and interrupts
        //
        VECTOR __stack_end__
        VECTOR Reset_Handler
        EXC_HANDLER NMI_Handler
        VECTOR HardFault_Handler
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
        ISR_RESERVED
        EXC_HANDLER SVC_Handler
        ISR_RESERVED
        ISR_RESERVED
        EXC_HANDLER PendSV_Handler
        EXC_HANDLER SysTick_Handler
        //
        // External interrupts
        //
#ifndef __NO_EXTERNAL_INTERRUPTS
        ISR_HANDLER INT00_IRQHandler         
        ISR_HANDLER INT01_IRQHandler         
        ISR_HANDLER INT02_IRQHandler         
        ISR_HANDLER INT03_IRQHandler         
        ISR_HANDLER INT04_IRQHandler         
        ISR_HANDLER INT05_IRQHandler         
        ISR_HANDLER INT06_IRQHandler         
        ISR_HANDLER INT07_IRQHandler         
        ISR_HANDLER INT08_IRQHandler         
        ISR_HANDLER INT09_IRQHandler         
        ISR_HANDLER INT10_IRQHandler         
        ISR_HANDLER INT11_IRQHandler         
        ISR_HANDLER INT12_IRQHandler         
        ISR_HANDLER INT13_IRQHandler         
        ISR_HANDLER INT14_IRQHandler         
        ISR_HANDLER INT15_IRQHandler         
        ISR_HANDLER INT16_IRQHandler         
        ISR_HANDLER INT17_18_32_33_IRQHandler
        ISR_HANDLER INT19_22_IRQHandler      
        ISR_HANDLER INT23_26_IRQHandler      
        ISR_HANDLER INT27_28_IRQHandler      
        ISR_HANDLER INT29_IRQHandler         
        ISR_HANDLER INT30_31_IRQHandler      
        ISR_HANDLER INTEMG0_IRQHandler       
        ISR_HANDLER INTOVV0_IRQHandler       
        ISR_HANDLER INTPMD0_IRQHandler       
        ISR_HANDLER INTENC00_IRQHandler      
        ISR_HANDLER INTENC01_IRQHandler      
        ISR_HANDLER INTADAPDA_IRQHandler     
        ISR_HANDLER INTADAPDB_IRQHandler     
        ISR_HANDLER INTADACP0_IRQHandler     
        ISR_HANDLER INTADACP1_IRQHandler     
        ISR_HANDLER INTADATRG_IRQHandler     
        ISR_HANDLER INTADASGL_IRQHandler     
        ISR_HANDLER INTADACNT_IRQHandler     
        ISR_HANDLER INTT0RX_IRQHandler       
        ISR_HANDLER INTT0TX_IRQHandler       
        ISR_HANDLER INTT0ERR_IRQHandler      
        ISR_HANDLER INTT1RX_IRQHandler       
        ISR_HANDLER INTT1TX_IRQHandler       
        ISR_HANDLER INTT1ERR_IRQHandler      
        ISR_HANDLER INTT2RX_IRQHandler       
        ISR_HANDLER INTT2TX_IRQHandler       
        ISR_HANDLER INTT2ERR_IRQHandler      
        ISR_HANDLER INTT3RX_IRQHandler       
        ISR_HANDLER INTT3TX_IRQHandler       
        ISR_HANDLER INTT3ERR_IRQHandler      
        ISR_HANDLER INTT4RX_IRQHandler       
        ISR_HANDLER INTT4TX_IRQHandler       
        ISR_HANDLER INTT4ERR_IRQHandler      
        ISR_HANDLER INTI2CWUP_IRQHandler     
        ISR_HANDLER INTI2C0NST_IRQHandler    
        ISR_HANDLER INTI2C0ATX_IRQHandler    
        ISR_HANDLER INTI2C0BRX_IRQHandler    
        ISR_HANDLER INTI2C0NA_IRQHandler     
        ISR_HANDLER INTI2C1NST_IRQHandler    
        ISR_HANDLER INTI2C1ATX_IRQHandler    
        ISR_HANDLER INTI2C1BRX_IRQHandler    
        ISR_HANDLER INTI2C1NA_IRQHandler     
        ISR_HANDLER INTI2C2NST_IRQHandler    
        ISR_HANDLER INTI2C2ATX_IRQHandler    
        ISR_HANDLER INTI2C2BRX_IRQHandler    
        ISR_HANDLER INTI2C2NA_IRQHandler     
        ISR_HANDLER INTI2C3NST_IRQHandler    
        ISR_HANDLER INTI2C3ATX_IRQHandler    
        ISR_HANDLER INTI2C3BRX_IRQHandler    
        ISR_HANDLER INTI2C3NA_IRQHandler     
        ISR_HANDLER INTUART0RX_IRQHandler    
        ISR_HANDLER INTUART0TX_IRQHandler    
        ISR_HANDLER INTUART0ERR_IRQHandler   
        ISR_HANDLER INTUART1RX_IRQHandler    
        ISR_HANDLER INTUART1TX_IRQHandler    
        ISR_HANDLER INTUART1ERR_IRQHandler   
        ISR_HANDLER INTUART2RX_IRQHandler    
        ISR_HANDLER INTUART2TX_IRQHandler    
        ISR_HANDLER INTUART2ERR_IRQHandler   
        ISR_HANDLER INTUART3RX_IRQHandler    
        ISR_HANDLER INTUART3TX_IRQHandler    
        ISR_HANDLER INTUART3ERR_IRQHandler   
        ISR_HANDLER INTUART4RX_IRQHandler    
        ISR_HANDLER INTUART4TX_IRQHandler    
        ISR_HANDLER INTUART4ERR_IRQHandler   
        ISR_HANDLER INTUART5RX_IRQHandler    
        ISR_HANDLER INTUART5TX_IRQHandler    
        ISR_HANDLER INTUART5ERR_IRQHandler   
        ISR_HANDLER INTT32A00A_IRQHandler    
        ISR_HANDLER INTT32A00ACAP0_IRQHandler
        ISR_HANDLER INTT32A00ACAP1_IRQHandler
        ISR_HANDLER INTT32A00B_IRQHandler    
        ISR_HANDLER INTT32A00BCAP0_IRQHandler
        ISR_HANDLER INTT32A00BCAP1_IRQHandler
        ISR_HANDLER INTT32A00C_IRQHandler    
        ISR_HANDLER INTT32A00CCAP0_IRQHandler
        ISR_HANDLER INTT32A00CCAP1_IRQHandler
        ISR_HANDLER INTT32A01A_IRQHandler    
        ISR_HANDLER INTT32A01ACAP0_IRQHandler
        ISR_HANDLER INTT32A01ACAP1_IRQHandler
        ISR_HANDLER INTT32A01B_IRQHandler    
        ISR_HANDLER INTT32A01BCAP0_IRQHandler
        ISR_HANDLER INTT32A01BCAP1_IRQHandler
        ISR_HANDLER INTT32A01C_IRQHandler    
        ISR_HANDLER INTT32A01CCAP0_IRQHandler
        ISR_HANDLER INTT32A01CCAP1_IRQHandler
        ISR_HANDLER INTT32A02A_IRQHandler    
        ISR_HANDLER INTT32A02ACAP0_IRQHandler
        ISR_HANDLER INTT32A02ACAP1_IRQHandler
        ISR_HANDLER INTT32A02B_IRQHandler    
        ISR_HANDLER INTT32A02BCAP0_IRQHandler
        ISR_HANDLER INTT32A02BCAP1_IRQHandler
        ISR_HANDLER INTT32A02C_IRQHandler    
        ISR_HANDLER INTT32A02CCAP0_IRQHandler
        ISR_HANDLER INTT32A02CCAP1_IRQHandler
        ISR_HANDLER INTT32A03A_IRQHandler    
        ISR_HANDLER INTT32A03ACAP0_IRQHandler
        ISR_HANDLER INTT32A03ACAP1_IRQHandler
        ISR_HANDLER INTT32A03B_IRQHandler    
        ISR_HANDLER INTT32A03BCAP0_IRQHandler
        ISR_HANDLER INTT32A03BCAP1_IRQHandler
        ISR_HANDLER INTT32A03C_IRQHandler    
        ISR_HANDLER INTT32A03CCAP0_IRQHandler
        ISR_HANDLER INTT32A03CCAP1_IRQHandler
        ISR_HANDLER INTT32A04A_IRQHandler    
        ISR_HANDLER INTT32A04ACAP0_IRQHandler
        ISR_HANDLER INTT32A04ACAP1_IRQHandler
        ISR_HANDLER INTT32A04B_IRQHandler    
        ISR_HANDLER INTT32A04BCAP0_IRQHandler
        ISR_HANDLER INTT32A04BCAP1_IRQHandler
        ISR_HANDLER INTT32A04C_IRQHandler    
        ISR_HANDLER INTT32A04CCAP0_IRQHandler
        ISR_HANDLER INTT32A04CCAP1_IRQHandler
        ISR_HANDLER INTT32A05A_IRQHandler    
        ISR_HANDLER INTT32A05ACAP0_IRQHandler
        ISR_HANDLER INTT32A05ACAP1_IRQHandler
        ISR_HANDLER INTT32A05B_IRQHandler    
        ISR_HANDLER INTT32A05BCAP0_IRQHandler
        ISR_HANDLER INTT32A05BCAP1_IRQHandler
        ISR_HANDLER INTT32A05C_IRQHandler    
        ISR_HANDLER INTT32A05CCAP0_IRQHandler
        ISR_HANDLER INTT32A05CCAP1_IRQHandler
        ISR_HANDLER INTT32A06A_IRQHandler    
        ISR_HANDLER INTT32A06ACAP0_IRQHandler
        ISR_HANDLER INTT32A06ACAP1_IRQHandler
        ISR_HANDLER INTT32A06B_IRQHandler    
        ISR_HANDLER INTT32A06BCAP0_IRQHandler
        ISR_HANDLER INTT32A06BCAP1_IRQHandler
        ISR_HANDLER INTT32A06C_IRQHandler    
        ISR_HANDLER INTT32A06CCAP0_IRQHandler
        ISR_HANDLER INTT32A06CCAP1_IRQHandler
        ISR_HANDLER INTT32A07A_IRQHandler    
        ISR_HANDLER INTT32A07ACAP0_IRQHandler
        ISR_HANDLER INTT32A07ACAP1_IRQHandler
        ISR_HANDLER INTT32A07B_IRQHandler    
        ISR_HANDLER INTT32A07BCAP0_IRQHandler
        ISR_HANDLER INTT32A07BCAP1_IRQHandler
        ISR_HANDLER INTT32A07C_IRQHandler    
        ISR_HANDLER INTT32A07CCAP0_IRQHandler
        ISR_HANDLER INTT32A07CCAP1_IRQHandler
        ISR_HANDLER INTPARI_IRQHandler       
        ISR_HANDLER INTDMAATC_IRQHandler     
        ISR_HANDLER INTDMAAERR_IRQHandler    
        ISR_HANDLER INTDMABTC_IRQHandler     
        ISR_HANDLER INTDMABERR_IRQHandler    
        ISR_HANDLER INTRTC_IRQHandler        
        ISR_HANDLER INTRMC0_IRQHandler       
        ISR_HANDLER INTFLCRDY_IRQHandler     
        ISR_HANDLER INTFLDRDY_IRQHandler     
        ISR_HANDLER INTLCDBUSF_IRQHandler    
        ISR_HANDLER INTLCDSTOP_IRQHandler    
        ISR_HANDLER INTUART6RX_IRQHandler    
        ISR_HANDLER INTUART6TX_IRQHandler    
        ISR_HANDLER INTUART6ERR_IRQHandler   
        ISR_HANDLER INTUART7RX_IRQHandler    
        ISR_HANDLER INTUART7TX_IRQHandler    
        ISR_HANDLER INTUART7ERR_IRQHandler   
#endif
        //
        .section .vectors, "ax"
_vectors_end:

#ifdef __VECTORS_IN_RAM
        //
        // Reserve space with the size of the vector table
        // in the designated RAM section.
        //
        .section .vectors_ram, "ax"
        .balign 512
        .global _vectors_ram

_vectors_ram:
        .space _vectors_end - _vectors, 0
#endif

/*********************************************************************
*
*  Dummy handler to be used for reserved interrupt vectors
*  and weak implementation of interrupts.
*
*/
        .section .init.Dummy_Handler, "ax"
        .thumb_func
        .weak Dummy_Handler
        .balign 2
Dummy_Handler:
        1: b 1b   // Endless loop


/*************************** End of file ****************************/
