/*********************************************************************
*               (c) SEGGER Microcontroller GmbH & Co. KG             *
*                        The Embedded Experts                        *
*                           www.segger.com                           *
**********************************************************************

-------------------------- END-OF-HEADER -----------------------------

File    : OS_TraceDemo.c
Purpose : .
*/

#include "RTOS.h"
#include "BSP.h"

/*********************************************************************
*
*       Static data
*
**********************************************************************
*/

static OS_STACKPTR int StackMain[128], StackLED[128];   /* Task stacks */
static OS_TASK         TCBMAIN, TCBLED;         /* Task-control-blocks */

volatile unsigned int _TestFunc0Cnt;
volatile unsigned int _TestFunc1Cnt;

/*********************************************************************
*
*       Types
*
**********************************************************************
*/

typedef void (*TestFunc) (void);

/*********************************************************************
*
*       Local functions
*
**********************************************************************
*/

/********************************************************************* 
*
*       _RunConditionalTest()
*
*  Function description
*    Demonstrate conditional execution.
*    Written in assembly language for deterministic behavior.
*/
static int _RunConditionalTest(int a) {
  int RetVal;
  //
  // RetVal = (a < 10) ? 1 : 0;
  //
  __asm volatile ("cmp %1, #9     \n\t"
                  "ite le         \n\t"
                  "ldrle %0, =#1  \n\t" // Executed when (a < 10)
                  "ldrgt %0, =#0  \n\t" // Executed otherwise
                  : "=r" (RetVal)       // Output Operands
                  : "r" (a)             // Input Operands
                  :                     // Clobbers
                  );                            
  return RetVal;
}

/********************************************************************* 
*
*       _RunConditionalBranchTest()
*
*  Function description
*    Demonstrate conditional branch execution.
*    Written in assembly language for deterministic behavior.
*/
static int _RunConditionalBranchTest(int a) {
  int RetVal;
  //
  // if (a < 10) {
  //   RetVal = 1;
  // } else {
  //   RetVal = 0;
  // }
  //
  __asm volatile ("cmp %1, #9     \n\t"
                  "bcc _Greater   \n\t"
                  "ldr %0, =#1    \n\t" // Executed when (a < 10)
                  "b _Return      \n\t"
                  "_Greater:      \n\t"
                  "ldr %0, =#0    \n\t" // Executed otherwise
                  "_Return:       \n\t"

                  : "=r" (RetVal)       // Output Operands
                  : "r" (a)             // Input Operands
                  :                     // Clobbers
                  );                            
  return RetVal;
}

/********************************************************************* 
*
*       _TestFunc0b()
*
*  Function description
*    Simply dummy function called via function pointer.
*/
static void _TestFunc0b(void) {
  _TestFunc0Cnt = 0;
}

/********************************************************************* 
*
*       _TestFunc0b()
*
*  Function description
*    Simply dummy function called via function pointer.
*/
static void _TestFunc0a(void) {
  _TestFunc0Cnt++;
}

/********************************************************************* 
*
*       _TestFunc0()
*
*  Function description
*    Simple dummy function using function pointers.
*/
static void _TestFunc0(void) {
  TestFunc pF;

  if (_TestFunc0Cnt < 10000) {
    pF = _TestFunc0a;
  } else {
    pF = _TestFunc0b;
  }
  pF();
}

/********************************************************************* 
*
*       _TestFunc1()
*
*  Function description
*    Simple dummy function.
*/
static void _TestFunc1(void) {
  if (_TestFunc1Cnt < 10000) {
    _TestFunc1Cnt++;
  } else {
    _TestFunc1Cnt = 0;
  }
}

/********************************************************************* 
*
*       HPTask()
*
*/
static void MainTask(void) {
  int Cnt;

  Cnt = 0;
  BSP_SetLED(0);
  while (1) {
    _RunConditionalTest(Cnt);
    _RunConditionalBranchTest(Cnt);
    _TestFunc0();
    _TestFunc1();
    Cnt++;
    OS_Delay(50);
  }
}

/********************************************************************* 
*
*       LPTask()
*
*/
static void LEDTask(void) {
  int On;
  //
  // Demonstrate different run counts.
  //
  On = 1;
  while (1) {
    //
    // The if statement is executed 9 times
    // while the else statement is executed 1 time.
    //
    if (On > 0) {
      BSP_ClrLED(1);
    } else {
      BSP_SetLED(1);
    }
    //
    // Multiple statements in a single line.
    // Both statements execute 5 times alternately.
    //
    (On > 5) ? BSP_SetLED(2) :  BSP_ClrLED(2);
    On = (On + 1) % 10;
    OS_Delay(200);
  }
}

/*********************************************************************
*
*       Global functions
*
**********************************************************************
*/
/********************************************************************* 
*
*       main()
*/
int main(void) {
  OS_IncDI();                      /* Initially disable interrupts  */
  OS_InitKern();                   /* Initialize OS                 */
  OS_InitHW();                     /* Initialize Hardware for OS    */
  BSP_Init();                      /* Initialize LED ports          */
  /* You need to create at least one task before calling OS_Start() */
  OS_CREATETASK(&TCBMAIN, "MainTask", MainTask, 100, StackMain);
  OS_CREATETASK(&TCBLED, "LED Task", LEDTask,   50, StackLED);
  OS_Start();                      /* Start multitasking            */
  return 0;
}

/****** End Of File *************************************************/
