/*********************************************************************
*                     SEGGER Microcontroller GmbH                    *
*                        The Embedded Experts                        *
**********************************************************************
*                                                                    *
*       (c) 1995 - 2021 SEGGER Microcontroller GmbH                  *
*                                                                    *
*       Internet: segger.com  Support: support_embos@segger.com      *
*                                                                    *
**********************************************************************
*                                                                    *
*       embOS * Real time operating system for microcontrollers      *
*                                                                    *
*       Please note:                                                 *
*                                                                    *
*       Knowledge of this file may under no circumstances            *
*       be used to write a similar product or a real-time            *
*       operating system for in-house use.                           *
*                                                                    *
*       Thank you for your fairness !                                *
*                                                                    *
**********************************************************************
*                                                                    *
*       OS version: V5.14.0.0                                        *
*                                                                    *
**********************************************************************

----------------------------------------------------------------------
File    : BSP.c
Purpose : BSP for Nordic nRF52 [Preview] Development Kit

Literature:
  [1]  http://infocenter.nordicsemi.com/topic/com.nordic.infocenter.nrf52/dita/nrf52/development/preview_dev_kit/hw_btns_leds.html
--------  END-OF-HEADER  ---------------------------------------------
*/

#include "BSP.h"

/*********************************************************************
*
*       Defines
*
**********************************************************************
*/
#define LED1_BIT       (17)  // LED1, Port 0 Pin 17
#define LED2_BIT       (18)  // LED2, Port 0 Pin 18
#define LED3_BIT       (19)  // LED3, Port 0 Pin 19
#define LED4_BIT       (20)  // LED4, Port 0 Pin 20

#define P0_BASE_ADDR   ((unsigned int)0x50000000)
#define P0_OUT_OFF     (0x504)
#define P0_PIN_CNF_OFF (0x700)
#define P0_OUT         (*(volatile unsigned int*)(P0_BASE_ADDR + P0_OUT_OFF))
#define LED1_CNF       (*(volatile unsigned int*)(P0_BASE_ADDR + P0_PIN_CNF_OFF + (4 * LED1_BIT)))
#define LED2_CNF       (*(volatile unsigned int*)(P0_BASE_ADDR + P0_PIN_CNF_OFF + (4 * LED2_BIT)))
#define LED3_CNF       (*(volatile unsigned int*)(P0_BASE_ADDR + P0_PIN_CNF_OFF + (4 * LED3_BIT)))
#define LED4_CNF       (*(volatile unsigned int*)(P0_BASE_ADDR + P0_PIN_CNF_OFF + (4 * LED4_BIT)))

/*********************************************************************
*
*       Global functions
*
**********************************************************************
*/

/*********************************************************************
*
*       BSP_Init()
*/
void BSP_Init(void) {
  //
  // CNF = dir out | input disconnected | pull disabled | drive S0S1 | sense disabled;
  //
  LED1_CNF = (1u << 0) | (1u << 1) | (0u << 2) | (0u << 8) | (0u << 16);
  LED2_CNF = (1u << 0) | (1u << 1) | (0u << 2) | (0u << 8) | (0u << 16);
  LED3_CNF = (1u << 0) | (1u << 1) | (0u << 2) | (0u << 8) | (0u << 16);
  LED4_CNF = (1u << 0) | (1u << 1) | (0u << 2) | (0u << 8) | (0u << 16);
  //
  // Initially clear LEDs, active low
  //
  P0_OUT |= (1u << LED1_BIT)| (1u << LED2_BIT) | (1u << LED3_BIT) | (1u << LED4_BIT);
}

/*********************************************************************
*
*       BSP_SetLED()
*/
void BSP_SetLED(int Index) {
  if (Index == 0) {
    P0_OUT &= ~(1u << LED1_BIT);  // Set LED1
  } else if (Index == 1) {
    P0_OUT &= ~(1u << LED2_BIT);  // Set LED2
  } else if (Index == 2) {
    P0_OUT &= ~(1u << LED3_BIT);  // Set LED3
  } else if (Index == 3) {
    P0_OUT &= ~(1u << LED4_BIT);  // Set LED4
  }
}

/*********************************************************************
*
*       BSP_ClrLED()
*/
void BSP_ClrLED(int Index) {
  if (Index == 0) {
    P0_OUT |= (1u << LED1_BIT);  // Clear LED1
  } else if (Index == 1) {
    P0_OUT |= (1u << LED2_BIT);  // Clear LED2
  } else if (Index == 2) {
    P0_OUT |= (1u << LED3_BIT);  // Clear LED3
  } else if (Index == 3) {
    P0_OUT |= (1u << LED4_BIT);  // Clear LED4
  }
}

/*********************************************************************
*
*       BSP_ToggleLED()
*/
void BSP_ToggleLED(int Index) {
  if (Index == 0) {
    if (P0_OUT & (1u << LED1_BIT)) {  // LED1 is switched off
      P0_OUT &= ~(1u << LED1_BIT);    // Set LED1
    } else {
      P0_OUT |=  (1u << LED1_BIT);    // Clear LED1
    }
  } else if (Index == 1) {
    if (P0_OUT & (1u << LED2_BIT)) {  // LED2 is switched off
      P0_OUT &= ~(1u << LED2_BIT);    // Set LED2
    } else {
      P0_OUT |=  (1u << LED2_BIT);    // Clear LED2
    }
  } else if (Index == 2) {
    if (P0_OUT & (1u << LED3_BIT)) {  // LED3 is switched off
      P0_OUT &= ~(1u << LED3_BIT);    // Set LED3
    } else {
      P0_OUT |=  (1u << LED3_BIT);    // Clear LED3
    }
  } else if (Index == 3) {
    if (P0_OUT & (1u << LED4_BIT)) {  // LED4 is switched off
      P0_OUT &= ~(1u << LED4_BIT);    // Set LED4
    } else {
      P0_OUT |=  (1u << LED4_BIT);    // Clear LED4
    }
  }
}

/****** End Of File *************************************************/
