/*
** ###################################################################
**     Processors:          MIMXRT685SFAWBR_cm33
**                          MIMXRT685SFFOB_cm33
**                          MIMXRT685SFVKB_cm33
**
**     Compilers:           GNU C Compiler
**                          IAR ANSI C/C++ Compiler for ARM
**                          Keil ARM C/C++ Compiler
**                          MCUXpresso Compiler
**
**     Reference manual:    MIMXRT685 User manual Rev. 0.95 11 November 2019
**     Version:             rev. 2.0, 2019-11-12
**     Build:               b191128
**
**     Abstract:
**         Provides a system configuration function and a global variable that
**         contains the system frequency. It configures the device and initializes
**         the oscillator (PLL) that is part of the microcontroller device.
**
**     Copyright 2016 Freescale Semiconductor, Inc.
**     Copyright 2016-2019 NXP
**     All rights reserved.
**
**     SPDX-License-Identifier: BSD-3-Clause
**
**     http:                 www.nxp.com
**     mail:                 support@nxp.com
**
**     Revisions:
**     - rev. 1.0 (2018-06-19)
**         Initial version.
**     - rev. 2.0 (2019-11-12)
**         Base on rev 0.95 RM (B0 Header)
**
** ###################################################################
*/

/*!
 * @file MIMXRT685S_cm33
 * @version 2.0
 * @date 2019-11-12
 * @brief Device specific configuration file for MIMXRT685S_cm33 (implementation
 *        file)
 *
 * Provides a system configuration function and a global variable that contains
 * the system frequency. It configures the device and initializes the oscillator
 * (PLL) that is part of the microcontroller device.
 */

#include <stdint.h>
#include "fsl_device_registers.h"

#define CLKCTL0_BASE_ADDR (0x40001000u)
#define CLKCTL0_SYSPLLCLKSEL (*(volatile unsigned int*)(CLKCTL0_BASE_ADDR + 0x200u))
#define CLKCTL0_SYSPLL0CTL0  (*(volatile unsigned int*)(CLKCTL0_BASE_ADDR + 0x204u))
#define CLKCTL0_SYSPLL0NUM   (*(volatile unsigned int*)(CLKCTL0_BASE_ADDR + 0x210u))
#define CLKCTL0_SYSPLL0DENOM (*(volatile unsigned int*)(CLKCTL0_BASE_ADDR + 0x214u))
#define CLKCTL0_SYSPLL0PFD   (*(volatile unsigned int*)(CLKCTL0_BASE_ADDR + 0x218u))
#define CLKCTL0_MAINCLKSELB  (*(volatile unsigned int*)(CLKCTL0_BASE_ADDR + 0x434u))

#define SYSTEM_IS_XIP_FLEXSPI()                                                                               \
    ((((uint32_t)SystemCoreClockUpdate >= 0x08000000U) && ((uint32_t)SystemCoreClockUpdate < 0x10000000U)) || \
     (((uint32_t)SystemCoreClockUpdate >= 0x18000000U) && ((uint32_t)SystemCoreClockUpdate < 0x20000000U)))

/* Get OSC clock from SYSOSC_BYPASS */
static uint32_t getOscClk(void)
{
  return (CLKCTL0->SYSOSCBYPASS == 0U) ? CLK_XTAL_OSC_CLK : ((CLKCTL0->SYSOSCBYPASS == 1U) ? CLK_EXT_CLKIN : 0U);
}

/* Get FFRO clock from FFROCTL0 setting */
static uint32_t getFFroFreq(void)
{
  uint32_t freq = 0U;

  switch (CLKCTL0->FFROCTL0 & CLKCTL0_FFROCTL0_TRIM_RANGE_MASK)
  {
    case CLKCTL0_FFROCTL0_TRIM_RANGE(0):
      freq = CLK_FRO_48MHZ;
      break;
    case CLKCTL0_FFROCTL0_TRIM_RANGE(3):
      freq = CLK_FRO_60MHZ;
      break;
    default:
      break;
  }
  return freq;
}

static void SetSysClock(void);

/* ----------------------------------------------------------------------------
   -- Core clock
   ---------------------------------------------------------------------------- */

uint32_t SystemCoreClock = DEFAULT_SYSTEM_CLOCK;

/* ----------------------------------------------------------------------------
   -- SystemInit()
   ---------------------------------------------------------------------------- */

__attribute__ ((weak)) void SystemInit (void) {
#if ((__FPU_PRESENT == 1) && (__FPU_USED == 1))
  SCB->CPACR |= ((3UL << 10*2) | (3UL << 11*2));    /* set CP10, CP11 Full Access in Secure mode */
  #if defined (__ARM_FEATURE_CMSE) && (__ARM_FEATURE_CMSE == 3U)
  SCB_NS->CPACR |= ((3UL << 10*2) | (3UL << 11*2));    /* set CP10, CP11 Full Access in Non-secure mode */
  #endif /* (__ARM_FEATURE_CMSE) && (__ARM_FEATURE_CMSE == 3U) */
#endif /* ((__FPU_PRESENT == 1) && (__FPU_USED == 1)) */

  SCB->CPACR |= ((3UL << 0*2) | (3UL << 1*2));    /* set CP0, CP1 Full Access in Secure mode (enable PowerQuad) */

#if defined (__ARM_FEATURE_CMSE) && (__ARM_FEATURE_CMSE == 3U)
  SCB_NS->CPACR |= ((3UL << 0*2) | (3UL << 1*2));    /* set CP0, CP1 Full Access in Non-secure mode (enable PowerQuad) */
#endif /* (__ARM_FEATURE_CMSE) && (__ARM_FEATURE_CMSE == 3U) */

  SCB->NSACR |= ((3UL << 0) | (3UL << 10));   /* enable CP0, CP1, CP10, CP11 Non-secure Access */

  SYSCTL0->DSPSTALL = SYSCTL0_DSPSTALL_DSPSTALL_MASK;

  if (SYSTEM_IS_XIP_FLEXSPI() && (CACHE64_POLSEL->POLSEL == 0)) /* Enable cache to accelerate boot. */
  {
    /* set command to invalidate all ways and write GO bit to initiate command */
    CACHE64->CCR = CACHE64_CTRL_CCR_INVW1_MASK | CACHE64_CTRL_CCR_INVW0_MASK;
    CACHE64->CCR |= CACHE64_CTRL_CCR_GO_MASK;
    /* Wait until the command completes */
    while (CACHE64->CCR & CACHE64_CTRL_CCR_GO_MASK)
    {
    }
    /* Enable cache, enable write buffer */
    CACHE64->CCR = (CACHE64_CTRL_CCR_ENWRBUF_MASK | CACHE64_CTRL_CCR_ENCACHE_MASK);

    /* Set whole FlexSPI0 space to write through. */
    CACHE64_POLSEL->REG0_TOP = 0x07FFFC00U;
    CACHE64_POLSEL->REG1_TOP = 0x0U;
    CACHE64_POLSEL->POLSEL = 0x1U;

    __ISB();
    __DSB();
  }
  SetSysClock();
  SystemInitHook();
}

/* ----------------------------------------------------------------------------
   -- SystemCoreClockUpdate()
   ---------------------------------------------------------------------------- */

void SystemCoreClockUpdate (void) {

  /* iMXRT6xx systemCoreClockUpdate */
  uint32_t freq = 0U;
  uint64_t freqTmp = 0U;

  switch ((CLKCTL0->MAINCLKSELB) & CLKCTL0_MAINCLKSELB_SEL_MASK)
  {
    case CLKCTL0_MAINCLKSELB_SEL(0): /* MAINCLKSELA clock */
      switch ((CLKCTL0->MAINCLKSELA) & CLKCTL0_MAINCLKSELA_SEL_MASK)
      {
        case CLKCTL0_MAINCLKSELA_SEL(0): /* FFRO clock (48/60m_irc) divider by 4 */
          freq = getFFroFreq() / 4;
          break;
        case CLKCTL0_MAINCLKSELA_SEL(1): /* OSC clock (clk_in) */
          freq = getOscClk();
          break;
        case CLKCTL0_MAINCLKSELA_SEL(2): /* Low Power Oscillator Clock (1m_lposc) */
          freq = CLK_LPOSC_1MHZ;
          break;
        case CLKCTL0_MAINCLKSELA_SEL(3): /* FFRO clock */
          freq = getFFroFreq();
          break;
        default:
            break;
      }
      break;
    case CLKCTL0_MAINCLKSELB_SEL(1): /* SFRO clock */
      freq = CLK_FRO_16MHZ;
      break;
    case CLKCTL0_MAINCLKSELB_SEL(2): /* Main System PLL clock */
      switch ((CLKCTL0->SYSPLL0CLKSEL) & CLKCTL0_SYSPLL0CLKSEL_SEL_MASK)
      {
        case CLKCTL0_SYSPLL0CLKSEL_SEL(0): /* SFRO clock */
          freq = CLK_FRO_16MHZ;
          break;
        case CLKCTL0_SYSPLL0CLKSEL_SEL(1): /* OSC clock (clk_in) */
          freq = getOscClk();
          break;
        case CLKCTL0_SYSPLL0CLKSEL_SEL(2): /* FFRO clock (48/60m_irc) divider by 2 */
          freq = getFFroFreq() / 2;
          break;
        default:
          break;
      }

      if (((CLKCTL0->SYSPLL0CTL0) & CLKCTL0_SYSPLL0CTL0_BYPASS_MASK) == 0U)
      {
        /* PLL output frequency = Fref * (DIV_SELECT + NUM/DENOM). */
        freqTmp = ((uint64_t)freq * ((uint64_t)(CLKCTL0->SYSPLL0NUM))) / ((uint64_t)(CLKCTL0->SYSPLL0DENOM));
        freq *= ((CLKCTL0->SYSPLL0CTL0) & CLKCTL0_SYSPLL0CTL0_MULT_MASK) >> CLKCTL0_SYSPLL0CTL0_MULT_SHIFT;
        freq += (uint32_t)freqTmp;
        freq = (uint64_t)freq * 18 /
               ((CLKCTL0->SYSPLL0PFD & CLKCTL0_SYSPLL0PFD_PFD0_MASK) >> CLKCTL0_SYSPLL0PFD_PFD0_SHIFT);
      }
      break;

    case CLKCTL0_MAINCLKSELB_SEL(3): /* RTC 32KHz clock */
        freq = CLK_RTC_32K_CLK;
        break;

    default:
        break;
  }

  SystemCoreClock = freq / ((CLKCTL0->SYSCPUAHBCLKDIV & 0xffU) + 1U);

}

/* ----------------------------------------------------------------------------
   -- SystemInitHook()
   ---------------------------------------------------------------------------- */

__attribute__ ((weak)) void SystemInitHook (void) {
  /* Void implementation of the weak function. */
}

/* ----------------------------------------------------------------------------
   -- SetSysClock()
   ---------------------------------------------------------------------------- */

void SetSysClock (void) {
  volatile int counter = 0;
  //
  // Check SYSPLL0
  //   clk_in * (MULT + (NUM / DENUM)) = 16Mhz * (16 + (1 / 1)) = 272Mhz
  //
  if ((CLKCTL0_SYSPLLCLKSEL & 0x7u) != 0x0u) {
    CLKCTL0_SYSPLLCLKSEL = (CLKCTL0_SYSPLLCLKSEL & ~(0x7));  // Select 16m_irc as input clock
  }
  if ((CLKCTL0_SYSPLL0CTL0 & 0xFF0003u) != 0x00100000u) {
    CLKCTL0_SYSPLL0CTL0 = (CLKCTL0_SYSPLL0CTL0
                        &  ~(1u      <<  0)  // Clear bypass
                        &  ~(1u      <<  1)  // Clear reset
                        &  ~(0xFFFFu << 16)) // Clear multiplier
                        |   (16u     << 16); // Set multiplier to 16
  }
  if ((CLKCTL0_SYSPLL0NUM != 0x2u) |  (CLKCTL0_SYSPLL0DENOM != 0x3u)) {
    CLKCTL0_SYSPLL0NUM   = 0x1u;
    CLKCTL0_SYSPLL0DENOM = 0x1u;
  }
  while (counter < 1000) {
    counter++;  // Wait for PLL to be stable
  }
  //
  // Check PFD0
  //   clk_sel * (18 / PFD0) = 272Mhz * (18 / 29) = ~169Mhz
  //
  if ((CLKCTL0_SYSPLL0PFD & 0xFF) != 0x4Cu) {
    CLKCTL0_SYSPLL0PFD |=  (1u << 7);                                    // Gate PFD0
    CLKCTL0_SYSPLL0PFD  =  (CLKCTL0_SYSPLL0PFD & ~(0x3F)) | (29u << 0);  // Set PFD0 to 28
    CLKCTL0_SYSPLL0PFD &= ~(1u << 7);                                    // Ungate PFD0
    while ((CLKCTL0_SYSPLL0PFD & (1u << 6)) == 0) {
      ;  // Wait for PFD0 to be ready
    }
  }
  //
  // Check MAINCLKSELB
  //
  if (CLKCTL0_MAINCLKSELB != 0x2u) {
    CLKCTL0_MAINCLKSELB = 0x2u;  // Set main clock to SYSPLL0_PFD0
  }
}