
/******************************************************************************
*
* Copyright (C) 2017 Texas Instruments Incorporated - http://www.ti.com/
*
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions
* are met:
*
*  Redistributions of source code must retain the above copyright
*  notice, this list of conditions and the following disclaimer.
*
*  Redistributions in binary form must reproduce the above copyright
*  notice, this list of conditions and the following disclaimer in the
*  documentation and/or other materials provided with the
*  distribution.
*
*  Neither the name of Texas Instruments Incorporated nor the names of
*  its contributors may be used to endorse or promote products derived
*  from this software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
* "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
* LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
* A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
* OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
* SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
* LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
* DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
* THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************/

#include <stdint.h>
#include <msp.h>

/*----------------------------------------------------------------------------
  Define clocks
 *----------------------------------------------------------------------------*/
#define __SYSTEM_CLOCK    (16000000ul)

/*---- Equals the parameter passed to SysCtlClockSet() API of Driverlib-------*/
// < 8000000ul> SYSCTL_XTAL_8MHZ
// <16000000ul> SYSCTL_XTAL_16MHZ
// <25000000ul> SYSCTL_XTAL_25MHZ
#define __SYSTEM_CLOCK_CRISTAL_FREQ (25000000ul)

/*----------------------------------------------------------------------------
  Clock Variable definitions
 *----------------------------------------------------------------------------*/
uint32_t SystemCoreClock = __SYSTEM_CLOCK; /*!< System Clock Frequency (Core Clock)*/

/*----------------------------------------------------------------------------
 Prototype definitions
 *----------------------------------------------------------------------------*/
static uint32_t _GetSystemClock(void);

/*----------------------------------------------------------------------------
  Clock functions
 *----------------------------------------------------------------------------*/
void SystemCoreClockUpdate (void)            /* Get Core Clock Frequency      */
{
    // Calculate the actual system clock as PSYSDIV is always div-by 2.
    SystemCoreClock = _GetSystemClock() / 2;
}

/**
 * Initialize the system
 *
 * @param  none
 * @return none
 *
 * @brief  Setup the microcontroller system.
 *         Initialize the System.
 */
void SystemInit (void)
{
  #if (__FPU_USED == 1)
    SCB->CPACR |= ((3UL << 10*2) |                 /* set CP10 Full Access */
                   (3UL << 11*2)  );               /* set CP11 Full Access */
  #endif

#ifdef UNALIGNED_SUPPORT_DISABLE
  SCB->CCR |= SCB_CCR_UNALIGN_TRP_Msk;
#endif

  // Calculate the actual system clock as PSYSDIV is always div-by 2.
  SystemCoreClock = _GetSystemClock() / 2;
}


static uint32_t _GetSystemClock(void)
{
    uint32_t clock_freq = 0, PLLFreq0 = 0, PLLFreq1 = 0, Osc = 0;
    uint32_t MInt = 0, MFrac = 0, NDiv = 0, QDiv = 0;

    // Read the RSCLKCFG register to determine if PLL is being used.
    if ((SYSCTL->RSCLKCFG & SYSCTL_RSCLKCFG_USEPLL) != SYSCTL_RSCLKCFG_USEPLL)
    {
        // return default value in case PLL is not being used.
        return __SYSTEM_CLOCK;
    }

    // Assume 25MHz default value. Need to be changed in case other crystal was used
    Osc = __SYSTEM_CLOCK_CRISTAL_FREQ;

    // Read the PLLFREQ0 and PLLFREQ1 registers to get information on the
    // MINT, MFRAC, N and Q values of the PLL
    PLLFreq0 = SYSCTL->PLLFREQ0;
    PLLFreq1 = SYSCTL->PLLFREQ1;

    MInt = (PLLFreq0 & SYSCTL_PLLFREQ0_MINT_M)   >> SYSCTL_PLLFREQ0_MINT_S;
    MFrac = (PLLFreq0 & SYSCTL_PLLFREQ0_MFRAC_M) >> SYSCTL_PLLFREQ0_MFRAC_S;
    NDiv = (PLLFreq1 & SYSCTL_PLLFREQ1_N_M) >> SYSCTL_PLLFREQ1_N_S;
    QDiv = (PLLFreq1 & SYSCTL_PLLFREQ1_Q_M) >> SYSCTL_PLLFREQ1_Q_S;

    // Calculate the VCO at the output of the PLL
    clock_freq = (Osc * MInt) + ((Osc * MFrac) / 1024);
    clock_freq /= ((NDiv + 1) * (QDiv + 1));

    return clock_freq;
}
