/*********************************************************************
*                     SEGGER Microcontroller GmbH                    *
*                        The Embedded Experts                        *
**********************************************************************
*                                                                    *
*       (c) 1995 - 2022 SEGGER Microcontroller GmbH                  *
*                                                                    *
*       Internet: segger.com  Support: support_embos@segger.com      *
*                                                                    *
**********************************************************************
*                                                                    *
*       embOS-Ultra * Real time operating system                     *
*                                                                    *
*       Please note:                                                 *
*                                                                    *
*       Knowledge of this file may under no circumstances            *
*       be used to write a similar product or a real-time            *
*       operating system for in-house use.                           *
*                                                                    *
*       Thank you for your fairness !                                *
*                                                                    *
**********************************************************************
*                                                                    *
*       OS version: V5.16.0.0                                        *
*                                                                    *
**********************************************************************

-------------------------- END-OF-HEADER -----------------------------
File    : OS_StartLEDPWM.c
Purpose : embOS sample program running two simple tasks, each toggling
          an LED of the target hardware (as configured in BSP.c).
          One task performs a gradual pulse-width modulation on one of the LEDs.
          The other task toggles a second LED at a constant rate.
*/

#include "RTOS.h"
#include "BSP.h"

static OS_STACKPTR int StackHP[128], StackLP[128];  // Task stacks
static OS_TASK         TCBHP, TCBLP;                // Task control blocks

#define PWM_FREQ                (100u)                         // 100 Hz PWM frequency
#define PWM_PERIOD              (1000000u / PWM_FREQ)          // PWM period in microseconds (100 Hz => 10,000 microseconds)
#define PWM_RESOLUTION          (100u)                         // PWM resolution
#define PWM_RESOLUTION_PERIOD   (PWM_PERIOD / PWM_RESOLUTION)  // PWM resolution period in microseconds (100 microseconds)

static void HPTask(void) {
  unsigned int DutyCycle;

  while (1) {
    //
    // Gradually increase duty cycle from 1 to 100 percent to brighten the LED
    //
    for (DutyCycle = 1u; DutyCycle < 100u; DutyCycle += 2u) {
      BSP_SetLED(0);
      OS_TASK_Delay_us(DutyCycle * PWM_RESOLUTION_PERIOD);
      BSP_ClrLED(0);
      OS_TASK_Delay_us(PWM_PERIOD - (DutyCycle * PWM_RESOLUTION_PERIOD));
    }
    //
    // Gradually decrease duty cycle from 100 to 1 percent to darken the LED
    //
    for (DutyCycle = 100u; DutyCycle > 1u; DutyCycle -= 2u) {
      BSP_SetLED(0);
      OS_TASK_Delay_us(DutyCycle * PWM_RESOLUTION_PERIOD);
      BSP_ClrLED(0);
      OS_TASK_Delay_us(PWM_PERIOD - (DutyCycle * PWM_RESOLUTION_PERIOD));
    }
  }
}

static void LPTask(void) {
  while (1) {
    BSP_ToggleLED(1);
    OS_TASK_Delay_ms(200u);
  }
}

/*********************************************************************
*
*       main()
*/
int main(void) {
  OS_Init();    // Initialize embOS
  OS_InitHW();  // Initialize required hardware
  BSP_Init();   // Initialize LED ports
  OS_TASK_CREATE(&TCBHP, "HP Task", 100, HPTask, StackHP);
  OS_TASK_CREATE(&TCBLP, "LP Task",  50, LPTask, StackLP);
  OS_Start();   // Start embOS
  return 0;
}

/*************************** End of file ****************************/
