/*********************************************************************
*                     SEGGER Microcontroller GmbH                    *
*                        The Embedded Experts                        *
**********************************************************************
*                                                                    *
*       (c) 1995 - 2022 SEGGER Microcontroller GmbH                  *
*                                                                    *
*       Internet: segger.com  Support: support_embos@segger.com      *
*                                                                    *
**********************************************************************
*                                                                    *
*       embOS-Ultra * Real time operating system                     *
*                                                                    *
*       Please note:                                                 *
*                                                                    *
*       Knowledge of this file may under no circumstances            *
*       be used to write a similar product or a real-time            *
*       operating system for in-house use.                           *
*                                                                    *
*       Thank you for your fairness !                                *
*                                                                    *
**********************************************************************
*                                                                    *
*       OS version: V5.16.0.0                                        *
*                                                                    *
**********************************************************************

-------------------------- END-OF-HEADER -----------------------------
Purpose : This RTOSInit*.c demonstrates how low power modes can be used
          with embOS-Ultra.

Additional information:
          In order to be able to keep track of the passed time, a
          timer which is still clocked during low power modes is
          required. With STM32L073 the LPTIM1 can be clocked in all
          modes except the standby mode.

          If ENTER_LOW_POWER_MODE is set 0, no low power mode is
          entered.

          If ENTER_LOW_POWER_MODE is set 1, a low power mode is
          entered in OS_Idle(). Debugging the application is not
          possible if the processor enters the low power mode.

          If USE_LOW_POWER_TIMER is set to 0, TIM3 is used a continuous
          counter and the SysTick is used as a timer to generate interrupts.
          When ENTER_LOW_POWER_MODE is set to 1 and the application enters
          the low power mode, the clocks of these timers will be disabled.
          As a result, no timer interrupt will occur which wakes up the
          processor. Please note that other events still can wake up the
          processor. However, since the timers' clocks were disabled,
          the system time won't be correct anymore.

          If USE_LOW_POWER_TIMER is set to 1, LPTIM1 is used as a
          continuous counter and timer. Its clock is derived from the
          LSI which runs with ~37 kHz.
          When ENTER_LOW_POWER_MODE is set to 1 and the application enters
          the low power mode in OS_Idle(), the clock of this timer continues
          to run. Before entering the stop mode, interrupts will be disabled.
          When the timer interrupt wakes the processor up it will continue
          execution in OS_Idle(). This is required in order to clock
          the processor clock to 32 MHz again before executing the
          timer interrupt.
*/

#include "RTOS.h"
#include "SEGGER_SYSVIEW.h"
#include "stm32l0xx.h"

/*********************************************************************
*
*       Config
*
**********************************************************************
*/
#ifndef ENTER_LOW_POWER_MODE
#define ENTER_LOW_POWER_MODE  (1)
#endif

#ifndef USE_LOW_POWER_TIMER
#define USE_LOW_POWER_TIMER  (1)
#endif

/*********************************************************************
*
*       Defines
*
**********************************************************************
*/
#if (USE_LOW_POWER_TIMER == 1)
#define LPTIM_CLOCK_FREQUENCY   (37000u)  // LPTIM is configured to use LSI which is ~37kHz
#define LPTIM_PERIOD_DEDUCTION  (0x200u)  // Deduct ~13,8 milliseconds
#define LPTIM_PERIOD_MAX        (0xFFFFu)
#else
//
// Hardware timer (SysTick) runs at 32 MHz, while counter (TIM3) runs at 1 MHz.
// This means a factor of 32.
//
#define OS_TIMER_FACTOR     (32)
//
// Hardware timer has 24 bits, counter has 16.
// This means the maximal value that fits both the used hardware timer and the used counter is 0x0000FFFF (i.e. TIM_ARR_ARR_Msk).
// It also means we need to ensure an ISR is generated (and, thus, the scheduler is executed) before the counter completes a loop, and therefore need to deduct some additional cycles.
//
#define OS_TIMER_MAX_VALUE  (TIM_ARR_ARR_Msk)
#define OS_TIMER_DEDUCTION  (2000u)
#endif

/*********************************************************************
*
*       embOSView settings
*/
#ifndef OS_VIEW_IFSELECT
#if (ENTER_LOW_POWER_MODE == 1)
#define OS_VIEW_IFSELECT  OS_VIEW_DISABLED
#else
#define OS_VIEW_IFSELECT  OS_VIEW_IF_JLINK
#endif
#endif

#if (OS_VIEW_IFSELECT == OS_VIEW_IF_JLINK)
  #include "JLINKMEM.h"
#elif (OS_VIEW_IFSELECT == OS_VIEW_IF_UART)
  #include "BSP_UART.h"
  #define OS_UART      (0u)
  #define OS_BAUDRATE  (38400u)
#endif

/*********************************************************************
*
*       Static data
*
**********************************************************************
*/
#if (OS_VIEW_IFSELECT == OS_VIEW_IF_JLINK)
  const OS_U32 OS_JLINKMEM_BufferSize = 32u;  // Size of the communication buffer for JLINKMEM
#else
  const OS_U32 OS_JLINKMEM_BufferSize = 0u;   // Buffer not used
#endif

/*********************************************************************
*
*       Local functions
*
**********************************************************************
*/

#if (OS_VIEW_IFSELECT == OS_VIEW_IF_UART)
/*********************************************************************
*
*       _OS_OnRX()
*
*  Function description
*    Callback wrapper function for BSP UART module.
*/
static void _OS_OnRX(unsigned int Unit, unsigned char c) {
  OS_USE_PARA(Unit);
  OS_COM_OnRx(c);
}

/*********************************************************************
*
*       _OS_OnTX()
*
*  Function description
*    Callback wrapper function for BSP UART module.
*/
static int _OS_OnTX(unsigned int Unit) {
  OS_USE_PARA(Unit);
  return (int)OS_COM_OnTx();
}
#endif

/*********************************************************************
*
*       Global functions
*
**********************************************************************
*/

/*********************************************************************
*
*       BSP_OS_GetCycles()
*
*  Function description
*    Calculates the current time in timer cycles since reset.
*
*  Return value
*    Current time in timer cycles since reset.
*/
OS_U64 BSP_OS_GetCycles(void) {
  OS_U16 TimerCycles;
  OS_U16 CycleCntLow;
  OS_U64 CycleCntHigh;
  OS_U64 CycleCntCompare;
  OS_U64 CycleCnt64;

  do {
    CycleCnt64   = OS_TIME_GetTimestamp();  // Read current OS_Global.Time
    CycleCntLow  = (OS_U16)CycleCnt64;      // Extract lower 16 bits of OS_Global.Time
    CycleCntHigh = (CycleCnt64 >> 16);      // Extract higher 48 bits of OS_Global.Time
#if (USE_LOW_POWER_TIMER == 1)
    TimerCycles  = LPTIM1->CNT;             // Read current hardware CycleCount
#else
    TimerCycles  = TIM3->CNT;               // Read current hardware CycleCount
#endif
    //
    // CycleCnt64 and TimerCycles need to be retrieved "simultaneously" for the below calculation to work:
    // If the above code is interrupted after retrieving OS_Global.Time, but before reading the hardware counter,
    // it may happen that the hardware counter overflows multiple times before returning here.
    // We could then no longer accurately compare (TimerCycles < CycleCntLow) below.
    //
    // Hence, we check if the upper word of OS_Global.Time still matches the value we retrieved earlier and repeat
    // the process if it doesn't. This works because OS_Global.Time is updated once per timer interrupt, and the timer
    // interrupt must be more frequent than the hardware counter overflow, so any overflow has necessarily incremented the upper word.
    //
    CycleCntCompare = (OS_TIME_GetTimestamp() >> 16);
  } while (CycleCntHigh != CycleCntCompare);
  //
  // Calculate the current time in timer cycles since reset.
  // This is done by simply copying the 16 bit hardware time stamp into the lower 16 bits of OS_Global.Time, while
  // the upper 48 bits of OS_Global.Time count the number of overflows of the hardware counter.
  // Therefore, we compare the lower 16 bits of OS_Global.Time to the newly retrieved hardware time stamp:
  //
  if (TimerCycles < CycleCntLow) {
    //
    // The hardware counter has overflown and we must increment the upper 48 bits of OS_Global.Time.
    //
    CycleCnt64 = ((OS_U64)(CycleCntHigh + 1u) << 16) + TimerCycles;
  } else {
    //
    // The hardware counter has not overflown and we do not perform any additional action.
    //
    CycleCnt64 = ((OS_U64)CycleCntHigh << 16) + TimerCycles;
  }
  return CycleCnt64;
}

/*********************************************************************
*
*       BSP_OS_StartTimer()
*
*  Function description
*    Program the hardware timer required for embOS's time base.
*    The used hardware timer must generate an interrupt before the used counter completes a full loop.
*
*  Parameters
*    Cycles: The amount of cycles after which the hardware timer shall generate an interrupt.
*/
void BSP_OS_StartTimer(OS_U32 Cycles) {
#if (USE_LOW_POWER_TIMER == 1)
#if (OS_VIEW_IFSELECT == OS_VIEW_IF_JLINK)
  //
  // If embOSView via J-Link is selected, JLINKMEM_Process() needs to be executed periodically by the application.
  // In this exemplary BSP implementation, JLINKMEM_Process() is called from the SysTick_Handler().
  // We therefore limit the system tick to occur after a maximum of 10 milliseconds,
  // guaranteeing JLINKMEM_Process() is executed at least every 10 milliseconds.
  //
  if (Cycles > (LPTIM_CLOCK_FREQUENCY / 100u)) {
    Cycles = LPTIM_CLOCK_FREQUENCY / 100u;
  }
#endif
  //
  // Truncate the maximum timer period, so that there's enough time for the
  // system to record the current time after the timer interrupt occurred.
  // Else, it might be that a full round of the counter gets lost.
  //
  if (Cycles > (LPTIM_PERIOD_MAX - LPTIM_PERIOD_DEDUCTION)) {
    Cycles = LPTIM_PERIOD_MAX - LPTIM_PERIOD_DEDUCTION;
  }
  //
  // If Cycles is less than 4, trigger the interrupt by setting
  // it pending. Else, it might be that the compare event is missed
  // if the compare value is too close to the counter value.
  //
  if (Cycles < 4) {
    NVIC_SetPendingIRQ(LPTIM1_IRQn);
    //
    // Set compare register to max period, as it isn't required for
    // the next LPTIM interrupt, because we set it already pending.
    //
    Cycles = LPTIM_PERIOD_MAX - LPTIM_PERIOD_DEDUCTION;
  }
  //
  // Calculate and set new timestamp for the compare register
  //
  LPTIM1->ICR = LPTIM_ICR_CMPOKCF;                                               // Clear "Compare register update OK" flag
  LPTIM1->CMP = (LPTIM1->CNT + Cycles) & 0xFFFFu;                                // Write new compare value
  while ((LPTIM1->ISR & LPTIM_ISR_CMPOK) == 0);                                  // Wait until update of compare register finished
  LPTIM1->ICR = LPTIM_ICR_CMPOKCF;                                               // Clear "Compare register update OK" flag
#else
  SysTick->CTRL = SysTick_CTRL_CLKSOURCE_Msk
                | SysTick_CTRL_TICKINT_Msk;                                      // Disable SysTick
  if (Cycles > ((OS_TIMER_MAX_VALUE - OS_TIMER_DEDUCTION) / OS_TIMER_FACTOR)) {  // Check if hardware timer is able to count that far...
    Cycles  = OS_TIMER_MAX_VALUE - OS_TIMER_DEDUCTION;                           // ...if not, count as far as possible.
  } else {
    Cycles *= OS_TIMER_FACTOR;                                                   // ...otherwise, count to the given value.
  }
#if (OS_VIEW_IFSELECT == OS_VIEW_IF_JLINK)
  //
  // If embOSView via J-Link is selected, JLINKMEM_Process() needs to be executed periodically by the application.
  // In this exemplary BSP implementation, JLINKMEM_Process() is called from the SysTick_Handler().
  // We therefore limit the system tick to occur after a maximum of 10 milliseconds,
  // guaranteeing JLINKMEM_Process() is executed at least every 10 milliseconds.
  //
  if (Cycles > (SystemCoreClock / 100u)) {
    Cycles = SystemCoreClock / 100u;
  }
#endif
  SysTick->LOAD = Cycles;                                                        // Set reload register
  SysTick->VAL  = 0u;                                                            // Load the SysTick Counter Value
  SysTick->CTRL = SysTick_CTRL_ENABLE_Msk
                | SysTick_CTRL_CLKSOURCE_Msk
                | SysTick_CTRL_TICKINT_Msk;                                      // Enable SysTick
  SysTick->LOAD = 0u;                                                            // Make sure we receive one interrupt only by clearing the reload value
#endif
}

#if (USE_LOW_POWER_TIMER == 1)
/*********************************************************************
*
*       LPTIM1_IRQHandler()
*
*  Function description
*    This is the hardware timer exception handler.
*/
void LPTIM1_IRQHandler(void);
void LPTIM1_IRQHandler(void) {
  LPTIM1->ICR = LPTIM_ICR_CMPMCF;
  OS_INT_EnterNestable();
  OS_TICK_Handle();
#if (OS_VIEW_IFSELECT == OS_VIEW_IF_JLINK)
  JLINKMEM_Process();
#endif
  OS_INT_LeaveNestable();
}
#else
/*********************************************************************
*
*       SysTick_Handler()

*
*  Function description
*    This is the hardware timer exception handler.
*/
void SysTick_Handler(void) {
  SysTick->CTRL = SysTick_CTRL_CLKSOURCE_Msk
                | SysTick_CTRL_TICKINT_Msk;  // Disable SysTick
  OS_INT_EnterNestable();
  OS_TICK_Handle();
#if (OS_VIEW_IFSELECT == OS_VIEW_IF_JLINK)
  JLINKMEM_Process();
#endif
  OS_INT_LeaveNestable();
}
#endif

/*********************************************************************
*
*       OS_InitHW()
*
*  Function description
*    Initialize the hardware required for embOS to run.
*/
void OS_InitHW(void) {
  OS_INT_IncDI();
  //
  // Initialize NVIC vector table offset register if applicable for this device.
  // Might be necessary for RAM targets or application not running from 0x00.
  //
#if (defined(__VTOR_PRESENT) && (__VTOR_PRESENT == 1))
  SCB->VTOR = (OS_U32)&__Vectors;
#endif
  //
  // Enable instruction and data cache if applicable for this device
  //
#if (defined(__ICACHE_PRESENT) && (__ICACHE_PRESENT == 1))
  SCB_EnableICache();
#endif
#if (defined(__DCACHE_PRESENT) && (__DCACHE_PRESENT == 1))
  SCB_EnableDCache();
#endif
  //
  // Inform embOS about the frequency of the counter
  //
  {
    SystemCoreClockUpdate();
#if (USE_LOW_POWER_TIMER == 1)
    OS_SYSTIMER_CONFIG SysTimerConfig = { LPTIM_CLOCK_FREQUENCY };
#else
    OS_SYSTIMER_CONFIG SysTimerConfig = { (SystemCoreClock / OS_TIMER_FACTOR) };
#endif
    OS_TIME_ConfigSysTimer(&SysTimerConfig);
  }
#if (USE_LOW_POWER_TIMER == 1)
  //
  // Initialize LPTIM1
  //
  SCB->SCR        = SCB_SCR_SLEEPDEEP_Msk;
  RCC->APB2ENR   |= RCC_APB2ENR_DBGEN;
  DBGMCU->APB1FZ  = DBGMCU_APB1_FZ_DBG_LPTIMER_STOP;             // Stop LPTIM counter on halt
  RCC->CSR        = RCC_CSR_LSION;                               // Enable internal low-speed oscillator LSI
  while ((RCC->CSR & RCC_CSR_LSIRDY) == 0);                      // Wait for LSI to stabilize
  RCC->CCIPR     |= (1u << RCC_CCIPR_LPTIM1SEL_Pos);             // Select LSI for LPTIM
  RCC->APB1ENR   |= RCC_APB1ENR_LPTIM1EN;                        // Enable LPTIM clock
  RCC->APB1SMENR |= RCC_APB1SMENR_LPTIM1SMEN;                    // Enable LPTIM clock
  LPTIM1->ICR     = LPTIM1->ISR;                                 // Clear all status flags
  LPTIM1->CFGR    = (0u << 9);                                   // PRESC: Clock prescaler: Divide by one. LPTIM_CR_ENABLE may not be set.
  LPTIM1->IER     = LPTIM_IER_CMPMIE;                            // Enable compare match interrupt. LPTIM_CR_ENABLE may not be set.
  LPTIM1->CR      = LPTIM_CR_ENABLE;                             // Enable LPTIM. Required for writing compare register.
  LPTIM1->ICR     = LPTIM_ICR_ARROKCF;                           // Clear "Autoreload register update OK" flag
  LPTIM1->ARR     = 0xFFFFu;                                     // Write compare register. LPTIM_CR_ENABLE must be set.
  while ((LPTIM1->ISR & LPTIM_ISR_ARROK) == 0);                  // Wait until update finished
  LPTIM1->ICR     = LPTIM_ICR_ARROKCF;                           // Clear "Autoreload register update OK" flag
  BSP_OS_StartTimer(0xFFFFFFFFu);                                // Initialize compare register. LPTIM_CR_ENABLE must be set.
  LPTIM1->CR      = LPTIM_CR_ENABLE                              // Keep enabled
                  | LPTIM_CR_CNTSTRT;                            // Start counting
  NVIC_SetPriority(LPTIM1_IRQn, (1u << __NVIC_PRIO_BITS) - 2u);  // Set the priority higher than the PendSV priority
  NVIC_EnableIRQ(LPTIM1_IRQn);                                   // Enable interrupt
#else
  //
  // Start the counter (here: TIM3)
  //
  RCC->APB1ENR |= RCC_APB1ENR_TIM3EN_Msk;  // Enable Timer clock
  TIM3->ARR     = TIM_ARR_ARR_Msk;         // Set upper boundary for this timer to count towards
  TIM3->PSC     = OS_TIMER_FACTOR;         // This is optional; we could use any other prescaler. The higher the prescaler, the longer periods are allowed between tick interrupts.
  TIM3->CR1    |= TIM_CR1_CEN_Msk;         // Enable the timer
  //
  // Start the hardware timer (here: SysTick)
  //
  NVIC_SetPriority(SysTick_IRQn, (1u << __NVIC_PRIO_BITS) - 2u);  // Set the priority higher than the PendSV priority
  BSP_OS_StartTimer(0xFFFFFFFFu);
#endif
  //
  // Configure and initialize SEGGER SystemView
  //
#if (OS_SUPPORT_PROFILE != 0)
  SEGGER_SYSVIEW_Conf();
#endif
  //
  // Initialize communication for embOSView
  //
#if (OS_VIEW_IFSELECT == OS_VIEW_IF_JLINK)
  JLINKMEM_SetpfOnRx(OS_COM_OnRx);
  JLINKMEM_SetpfOnTx(OS_COM_OnTx);
  JLINKMEM_SetpfGetNextChar(OS_COM_GetNextChar);
#elif (OS_VIEW_IFSELECT == OS_VIEW_IF_UART)
  BSP_UART_Init(OS_UART, OS_BAUDRATE, BSP_UART_DATA_BITS_8, BSP_UART_PARITY_NONE, BSP_UART_STOP_BITS_1);
  BSP_UART_SetReadCallback(OS_UART, _OS_OnRX);
  BSP_UART_SetWriteCallback(OS_UART, _OS_OnTX);
#endif
  OS_INT_DecRI();
}

/*********************************************************************
*
*       OS_Idle()
*
*  Function description
*    This code is executed whenever no task, software timer, or
*    interrupt is ready for execution.
*    It may be used to e.g. enter a low power mode of the device.
*
*  Additional information
*    The idle loop does not have a stack of its own, therefore no
*    functionality should be implemented that relies on the stack
*    to be preserved.
*    The idle loop can be exited only when an embOS interrupt causes
*    a context switch (e.g. after expiration of a task timeout),
*    hence embOS interrupts must not permanently be disabled.
*/
void OS_Idle(void) {  // Idle loop: No task is ready to execute
  while (1) {
#if (ENTER_LOW_POWER_MODE == 1)
    //
    // Enter low power mode.
    // Disable interrupts, so that the application continues execution in
    // OS_Idle() after the processor woke up due to an interrupt event in
    // order to configure the processor clock to 32 MHz again.
    //
#if (USE_LOW_POWER_TIMER == 1)
    OS_U16 CyclesLeft;

    //
    // Enter low power mode only if at least 1ms is spent in OS_Idle().
    //
    CyclesLeft = (OS_U16)(LPTIM1->CMP - LPTIM1->CNT);
    if (CyclesLeft >= (LPTIM_CLOCK_FREQUENCY / 1000))
#endif
    {
      OS_INT_DisableAll();
      HAL_PWR_EnterSTOPMode(PWR_LOWPOWERREGULATOR_ON, PWR_SLEEPENTRY_WFI);
      SetSysClock();
      OS_INT_EnableAll();
    }
#endif
  }
}

/*********************************************************************
*
*       Optional communication with embOSView
*
**********************************************************************
*/

/*********************************************************************
*
*       OS_COM_Send1()
*
*  Function description
*    Sends one character.
*/
void OS_COM_Send1(OS_U8 c) {
#if (OS_VIEW_IFSELECT == OS_VIEW_IF_JLINK)
  JLINKMEM_SendChar(c);
#elif (OS_VIEW_IFSELECT == OS_VIEW_IF_UART)
  BSP_UART_Write1(OS_UART, c);
#elif (OS_VIEW_IFSELECT == OS_VIEW_DISABLED)
  OS_USE_PARA(c);          // Avoid compiler warning
  OS_COM_ClearTxActive();  // Let embOS know that Tx is not busy
#endif
}

/*************************** End of file ****************************/
