/*********************************************************************
*                    SEGGER Microcontroller GmbH                     *
*       Solutions for real time microcontroller applications         *
**********************************************************************
*                                                                    *
*            (c) 1995 - 2024 SEGGER Microcontroller GmbH             *
*                                                                    *
*       www.segger.com     Support: support@segger.com               *
*                                                                    *
**********************************************************************
*                                                                    *
* All rights reserved.                                               *
*                                                                    *
* * This software may in its unmodified form be freely redistributed *
*   in source form.                                                  *
* * The source code may be modified, provided the source code        *
*   retains the above copyright notice, this list of conditions and  *
*   the following disclaimer.                                        *
* * Modified versions of this software in source or linkable form    *
*   may not be distributed without prior consent of SEGGER.          *
* * This software may only be used for communication with SEGGER     *
*   J-Link debug probes.                                             *
*                                                                    *
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND             *
* CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,        *
* INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF           *
* MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE           *
* DISCLAIMED. IN NO EVENT SHALL SEGGER Microcontroller BE LIABLE FOR *
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR           *
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT  *
* OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;    *
* OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF      *
* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT          *
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE  *
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH   *
* DAMAGE.                                                            *
*                                                                    *
**********************************************************************
---------------------------END-OF-HEADER------------------------------
  Sample implementation for using J-Scope with the RTT-Plus mode.
  Generates 3 sine waveforms and sends data to J-Scope application using RTT.
*/

#include <math.h>         // For sin()
#include <string.h>       // For memset()
#include "SEGGER_RTT.h"   // For SEGGER RTT

//
// J-Scope RTT configurable defines.
//
#define JS_RTT_PLUS_USE_TIMESTAMP (1)         // With or without application defined timestamps.
#define JS_RTT_PLUS_MaxNumVars    (10)        // Maximum number of variables that can be sampled.
#define JS_RTT_PLUS_NumTrys       (0x50000)   // Number of trys to read out the configuration data.

//
// Error messages.
//
#define JS_RTT_PLUS_TIMEOUT   "Timeout while waiting for config data!"
#define JS_RTT_PLUS_VARERROR  "Increase variable buffer!"

//
// J-Scope RTT non configurable defines.
//
#define JS_RTT_PLUS_SUCCESS   "\0"                      // Do not change J-Scope expects this sting otherwise it wont start sampling.
#if JS_RTT_PLUS_USE_TIMESTAMP == 1
  #define JS_RTT_PLUS_UP_NAME "JScope_Data_t"           // RTT UP Buffer name to sample with timestamps.
#else
  #define JS_RTT_PLUS_UP_NAME "JScope_Data"             // RTT UP Buffer name to sample without timestamps.
#endif
#define JS_RTT_PLUS_DOWN_NAME "JScope_Config"           // RTT Down Buffer name

//
// Sinus related defines.
//
#define PI 3.14159265358979323846f
#define SINUS_SAMPLE_DATA  500  // Number of Sinus samples per 90
#define SINUS_SCALE      12000  // SIN_MAX = sin(90) = 1 * SINUS_SCALE (PI/2*SINUS_SCALE)

//
// J-Scope RTT related variables.
//
typedef struct JS_RTT_PLUS_SYMBOL {  // Needs to be 64 Bit aligned.
  unsigned long long Addr;             // 64 Bit
  unsigned int       NumBytes;         // 32 Bit
} JS_RTT_PLUS_SYMBOL;

JS_RTT_PLUS_SYMBOL aVarData[JS_RTT_PLUS_MaxNumVars + 1];

int  JS_RTT_PLUS_NumVars;
char JS_RTT_PLUS_UpBuffer[2048];      // J-Scope RTT Up Buffer: Minimum size 1024 Bytes. Set higher for better performance.
char JS_RTT_PLUS_DownBuffer[32];      // J-Scope RTT Down Buffer: Minimum size 16 Bytes.
int  JS_RTT_PLUS_ChannelUp   = 1;     // J-Scope RTT Channel
int  JS_RTT_PLUS_ChannelDown = 2;     // J-Scope RTT Channel

//
// Static sinus lookup table 
//
static unsigned _aSin[SINUS_SAMPLE_DATA];
static unsigned _SinAShift;
static unsigned _SinBShift;
static unsigned _SinCShift;

int Sine1;                     // Sine wave 1 
int Sine2;                     // Sine wave 2 
int Sine3;                     // Sine wave 3 
int Timestamp;                 // Timestamp

/*********************************************************************
*
*       _CalcSinusData()
*
* Function description
*   Fills _aSin[SINUS_SAMPLE_DATA] with data values 
*  
*/
static void _CalcSinusData(void) {
  unsigned i;
  for (i = 0; i < SINUS_SAMPLE_DATA; i++) {
    _aSin[i] = (sin((float)i * PI / (2. * SINUS_SAMPLE_DATA)) * SINUS_SCALE); // Calcs sin data for 0-90? (0-PI/2)
  }
  _SinAShift = SINUS_SAMPLE_DATA * 0.00000f;
  _SinBShift = SINUS_SAMPLE_DATA * 1.33333f;
  _SinCShift = SINUS_SAMPLE_DATA * 2.66667f;
}

/*********************************************************************
*
*       _GetSinVal()
*
* Function description
*   Returns the corresponding y-value for the x-Value passed
*
*  Parameter
*    XVal - 0 <= XVal < (4 * SINUS_SAMPLE_DATA)
*/
static int _GetSinVal(unsigned XVal) {
  switch (XVal / SINUS_SAMPLE_DATA) {
  case 0:
    return _aSin[XVal];                                          // Values 0?-90?   = "normal"
  case 1:
    return _aSin[(((2 * SINUS_SAMPLE_DATA) - 1) - XVal)];        // Values 90?-180? = 0?-90? in reverse order (=> 89? <=> 91?; 30? <=> 150? etc)
  case 2:
    return _aSin[XVal - (2 * SINUS_SAMPLE_DATA)] * (-1);         //Values 180?-270? = 0-90? inverted
  case 3:
    return _aSin[(((4 * SINUS_SAMPLE_DATA) - 1) - XVal)] * (-1); //Values 270?-360? = 90?-180? inverted
  default:
    return 0;
  }
}

/*********************************************************************
*
*       _SineLUT()
*
* Function description
*   Updates sine values for three phases (Sine1, Sine2, Sine3) based on the current counter and phase shifts.
*
*/
static void _SineLUT(void) {
  static unsigned XCnt;

  Sine1 = _GetSinVal((XCnt + _SinAShift) % (4 * SINUS_SAMPLE_DATA));
  Sine2 = _GetSinVal((XCnt + _SinBShift) % (4 * SINUS_SAMPLE_DATA));
  Sine3 = _GetSinVal((XCnt + _SinCShift) % (4 * SINUS_SAMPLE_DATA));
  XCnt++;
}

/*********************************************************************
*
*       JS_RTT_PLUS_SendData()
*
* Function description
*   Sends values of the requested variables via RTT to J-Scope.
*
*  Parameter
*    Timestamp: Timestamp in μs (Only used when JS_RTT_PLUS_USE_TIMESTAMP is defined as 1).
*/
void JS_RTT_PLUS_SendData(int Timestamp) {
  char* pVarData;
  int   VarAddr;
  int   VarSize;
  int   i;
#if JS_RTT_PLUS_USE_TIMESTAMP == 1
  char  acVarData[(JS_RTT_PLUS_NumVars + 1) * 4];       // Data for each variable is max 4 bytes large + 4 bytes timestamp. 
#else
  char  acVarData[JS_RTT_PLUS_NumVars * 4];             // Data for each variable is max 4 bytes large. 
#endif
  // 
  // Read out the requested addresses and variable sizes, then save the values of the variables in the acVarData buffer.
  // Formatting of the data in the paVarData: <4 Byte Addr Var1> <4 Byte size Var1> <4 Byte Addr Var2> <4 Byte size Var2>.
  // Formatting of the data in the acVarData buffer: <4 Byte Timestamp (optional)> <Data var 1> <Data var 2> <Data var 3>. | The size of the individual variable data is based on the size of that variable.
  //
  if (JS_RTT_PLUS_NumVars == 0) {
    return;
  }
  pVarData = &acVarData[0];
#if JS_RTT_PLUS_USE_TIMESTAMP == 1
  *((int*)pVarData) = Timestamp;                          // Add the timestamp to the buffer.
  pVarData += 4;
#else
  (void)Timestamp;
#endif
  i = 0;
  while (i < JS_RTT_PLUS_NumVars) {
    VarSize = aVarData[i].NumBytes;                      // Get the variables size.
    VarAddr = aVarData[i].Addr;                          // Get the variables address.
    i++;
    switch (VarSize) {
    case 1:
      *pVarData = *((char*)VarAddr);                      // Get the value at the variables address as char and add it to the buffer.
      pVarData += 1;
    break;
    case 2:
      *((short*)pVarData) = *((short*)VarAddr);           // Get the value at the variables address as short and add it to the buffer.
      pVarData += 2;
    break;
    case 4:
      *((int*)pVarData) = *((int*)VarAddr);               // Get the value at the variables address as int and add it to the buffer.
      pVarData += 4;
    break;
    default:
    break;
    }
  }
  SEGGER_RTT_Write(JS_RTT_PLUS_ChannelUp, &acVarData[0], pVarData - &acVarData[0]);  // Send the variable data to J-Scope.
}

/*********************************************************************
*
*       JS_RTT_PLUS_WaitForConfig()
*
* Function description
*   Configures the RTT buffers and waits until the configuration data is sent by J-Scope.
*   When the maximum number of tries is reached, an error message is sent to J-Scope, and the program continues without sampling
*/
void JS_RTT_PLUS_WaitForConfig(void) {
  unsigned char* pVarData;
  int   NumBytes;
  int   NumBytesTotal;
  int   Cnt;
  char  acMsg[256];
  char* sMsg;
  //
  // Configure RTT buffer.
  //
  SEGGER_RTT_ConfigUpBuffer  (JS_RTT_PLUS_ChannelUp,   JS_RTT_PLUS_UP_NAME,   &JS_RTT_PLUS_UpBuffer[0],   sizeof(JS_RTT_PLUS_UpBuffer),   SEGGER_RTT_MODE_BLOCK_IF_FIFO_FULL);
  SEGGER_RTT_ConfigDownBuffer(JS_RTT_PLUS_ChannelDown, JS_RTT_PLUS_DOWN_NAME, &JS_RTT_PLUS_DownBuffer[0], sizeof(JS_RTT_PLUS_DownBuffer), SEGGER_RTT_MODE_BLOCK_IF_FIFO_FULL);
  //
  // Read out the configuration data from the RTT buffer, that is sent by J-Scope.
  //
  memset(&aVarData[0], 0, sizeof(aVarData));
  pVarData      = (unsigned char*)&aVarData[0];
  NumBytes      = 0;
  NumBytesTotal = 0;
  Cnt           = 0;
  while(1) {
    Cnt++;
    if (Cnt > JS_RTT_PLUS_NumTrys) {
      sMsg = JS_RTT_PLUS_TIMEOUT;
      JS_RTT_PLUS_NumVars = 0;
      break;                                                                                           // Error: Timeout.
    }
    NumBytes = SEGGER_RTT_Read(JS_RTT_PLUS_ChannelDown, pVarData, sizeof(aVarData) - NumBytesTotal);   // Read variable data.
    if (NumBytes <= 0) {
      continue;
    }
    pVarData      += NumBytes;
    NumBytesTotal += NumBytes;
    JS_RTT_PLUS_NumVars = NumBytesTotal / sizeof(JS_RTT_PLUS_SYMBOL);
    if (JS_RTT_PLUS_NumVars == 0) {
      continue;
    }
    //
    // Check if we have reached the end of the configuration data.
    //
    if (   (aVarData[JS_RTT_PLUS_NumVars - 1].NumBytes == 0)
        && (aVarData[JS_RTT_PLUS_NumVars - 1].Addr     == 0)) {
      sMsg = JS_RTT_PLUS_SUCCESS;
      JS_RTT_PLUS_NumVars -= 1;                                                                        // Ignore the last dummy entry that just indicates the end.
      break;
    }
    if (NumBytesTotal == sizeof(aVarData)) {
      sMsg = JS_RTT_PLUS_VARERROR;
      JS_RTT_PLUS_NumVars = 0;
      break;                                                                                           // Error: Variable buffer not large enough.
    }
  }
  //
  // Send the reply to J-Scope.
  //
  memset(&acMsg[0], 0, sizeof(acMsg));
  strcpy(&acMsg[0], sMsg);
  SEGGER_RTT_Write(JS_RTT_PLUS_ChannelUp, &acMsg[0], sizeof(acMsg));
}

int main(void) {
  JS_RTT_PLUS_WaitForConfig();
  _CalcSinusData();            // Init sinus lookup table.
  Timestamp = 0;
  while (1) {
    Timestamp += 100;          // Add 100μs each iteration.
    JS_RTT_PLUS_SendData(Timestamp);
    _SineLUT();
  }
}
