/*********************************************************************
*                    SEGGER Microcontroller GmbH                     *
*                        The Embedded Experts                        *
**********************************************************************
*                                                                    *
*            (c) 2014 - 2020 SEGGER Microcontroller GmbH             *
*                                                                    *
*           www.segger.com     Support: support@segger.com           *
*                                                                    *
**********************************************************************
*                                                                    *
* All rights reserved.                                               *
*                                                                    *
* Redistribution and use in source and binary forms, with or         *
* without modification, are permitted provided that the following    *
* conditions are met:                                                *
*                                                                    *
* - Redistributions of source code must retain the above copyright   *
*   notice, this list of conditions and the following disclaimer.    *
*                                                                    *
* - Neither the name of SEGGER Microcontroller GmbH                  *
*   nor the names of its contributors may be used to endorse or      *
*   promote products derived from this software without specific     *
*   prior written permission.                                        *
*                                                                    *
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND             *
* CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,        *
* INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF           *
* MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE           *
* DISCLAIMED.                                                        *
* IN NO EVENT SHALL SEGGER Microcontroller GmbH BE LIABLE FOR        *
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR           *
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT  *
* OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;    *
* OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF      *
* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT          *
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE  *
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH   *
* DAMAGE.                                                            *
*                                                                    *
**********************************************************************

-------------------------- END-OF-HEADER -----------------------------

File    : main.c
Purpose : Example application doing a CRC check using the STM32 CRC peripheral

*/

#include <stdio.h>
#include <stdlib.h>
#include "stm32f4xx.h"

#define FLASH_IMAGE_START 0x08000000
#define FLASH_IMAGE_END   0x0807FFFC
#define FLASH_APPLICATION_END (FLASH_IMAGE_END - 0x4)

/*********************************************************************
*
*       main()
*
*  Function description
*   Application entry point.
*/
int main(void) {
  int i;
  unsigned int NumItems;
  unsigned int* pData;
  unsigned int CRCResultHW;
  unsigned int CRCResult;
  unsigned int OldValue;
  
  i = 0;
  NumItems = (FLASH_APPLICATION_END - FLASH_IMAGE_START) / 4;
  pData = (unsigned int*)FLASH_IMAGE_START;    // points to start of Flash
  CRCResult = *(unsigned int*)FLASH_IMAGE_END; // Saves CRC value calculated by SEGGER Linker
  //
  // Config CRC Module
  //
  RCC->AHB1ENR |= RCC_AHB1ENR_CRCEN;
  CRC->CR |= CRC_CR_RESET;
  //
  // Calculate CRC with ST CRC unit over complete Flash area
  //
  do {
    CRC->DR = __REV(*pData);  // ST algorithm expects words in reversed order
    pData++;
  } while (NumItems--);
  CRCResultHW = CRC->DR;
  printf("Hardware calculated CRC over Flash is: 0x%X \n", CRCResultHW);
  printf("SEGGER Linker calculated CRC over Flash is: 0x%X\n", CRCResult);
  //
  // Compare with Linker result
  //
  if (CRCResult == CRCResultHW) {
    printf("Both CRC check sums are matching!\n");
  } else {
    printf("CRC check sums are not matching. Check parameters.\n");
  }
  do {
    i++;
  } while (1);
}

/*************************** End of file ****************************/
